<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Blyxxa PWA Module (Client)
 * 
 * Handles Mobile App features for the SaaS Client.
 * Uses settings from the Client Dashboard (to be added).
 */

class Blyxxa_PWA_Client {

    public function __construct() {
        // We will add settings to the existing Client Dashboard "Configuration" tab later.
        // For now, we use a separate option key 'blyxxa_client_pwa_settings'.
        
        add_action('wp_head', [$this, 'output_meta_tags']);
        add_action('init', [$this, 'add_endpoints']);
        add_action('template_redirect', [$this, 'serve_assets']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_pwa_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_pwa_scripts']); // Ensure styles load in admin
        add_action('wp_footer', [$this, 'render_app_bar']);
        add_action('admin_footer', [$this, 'render_app_bar']); // For Admin Dashboard
        
        // App Tool Registry
        add_action('add_meta_boxes', [$this, 'add_app_meta_box']);
        add_action('save_post', [$this, 'save_app_meta_box']);
    }

    public function add_app_meta_box() {
        add_meta_box('blyxxa_app_settings', '📱 ' . esc_html__('Blyxxa App Settings', 'blyxxa-core'), [$this, 'render_meta_box'], ['page', 'post'], 'side', 'high');
    }

    public function render_meta_box($post) {
        $show = get_post_meta($post->ID, 'blyxxa_app_show', true);
        $cat = get_post_meta($post->ID, 'blyxxa_app_category', true);
        $icon = get_post_meta($post->ID, 'blyxxa_app_icon', true);
        $cats = get_option('blyxxa_categories', ['General', 'Social Media', 'SEO', 'Copywriting', 'Email Marketing']);
        
        wp_nonce_field('blyxxa_app_save', 'blyxxa_app_nonce');
        ?>
        <p>
            <label>
                <input type="checkbox" name="blyxxa_app_show" value="1" <?php checked($show, '1'); ?>>
                <strong><?php esc_html_e('Show in Mobile App', 'blyxxa-core'); ?></strong>
            </label>
            <br>
            <span class="description" style="color:#64748b; font-size:12px; display:block; margin-top:3px;">
                <?php esc_html_e('Check this box to include this page in the "Tools" catalog of the mobile app.', 'blyxxa-core'); ?>
            </span>
        </p>
        <p>
            <label style="display:block; margin-bottom:5px; font-weight:600;"><?php esc_html_e('Category', 'blyxxa-core'); ?></label>
            <select name="blyxxa_app_category" style="width:100%;">
                <?php foreach($cats as $c): ?>
                    <option value="<?php echo esc_attr($c); ?>" <?php selected($cat, $c); ?>><?php echo esc_html($c); ?></option>
                <?php endforeach; ?>
            </select>
            <span class="description" style="color:#64748b; font-size:12px; display:block; margin-top:3px;">
                <?php esc_html_e('Select the category under which this tool will appear in the app.', 'blyxxa-core'); ?>
            </span>
        </p>
        <p>
            <label style="display:block; margin-bottom:5px; font-weight:600;"><?php esc_html_e('Icon / Emoji', 'blyxxa-core'); ?></label>
            <input type="text" name="blyxxa_app_icon" value="<?php echo esc_attr($icon); ?>" placeholder="e.g. 🚀" style="width:100%;">
            <span class="description" style="color:#64748b; font-size:12px; display:block; margin-top:3px;">
                <?php esc_html_e('Enter an emoji or a short text to represent this tool in the grid.', 'blyxxa-core'); ?>
            </span>
        </p>
        <?php
    }

    public function save_app_meta_box($post_id) {
        if (!isset($_POST['blyxxa_app_nonce']) || !wp_verify_nonce($_POST['blyxxa_app_nonce'], 'blyxxa_app_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        
        update_post_meta($post_id, 'blyxxa_app_show', isset($_POST['blyxxa_app_show']) ? '1' : '0');
        if(isset($_POST['blyxxa_app_category'])) update_post_meta($post_id, 'blyxxa_app_category', sanitize_text_field($_POST['blyxxa_app_category']));
        if(isset($_POST['blyxxa_app_icon'])) update_post_meta($post_id, 'blyxxa_app_icon', sanitize_text_field($_POST['blyxxa_app_icon']));
    }

    public function add_endpoints() {
        add_rewrite_rule('^blyxxa-manifest\.json$', 'index.php?blyxxa_pwa_manifest=1', 'top');
        add_rewrite_rule('^blyxxa-sw\.js$', 'index.php?blyxxa_pwa_sw=1', 'top');
        add_rewrite_tag('%blyxxa_pwa_manifest%', '([^&]+)');
        add_rewrite_tag('%blyxxa_pwa_sw%', '([^&]+)');

        if (!get_option('blyxxa_client_pwa_rules_flushed')) {
            flush_rewrite_rules();
            update_option('blyxxa_client_pwa_rules_flushed', true);
        }
    }

    public function serve_assets() {
        global $wp_query;
        
        // Serve Manifest
        if (isset($wp_query->query_vars['blyxxa_pwa_manifest'])) {
            $this->render_manifest();
        }

        // Serve Service Worker
        if (isset($wp_query->query_vars['blyxxa_pwa_sw'])) {
            $this->render_service_worker();
        }
    }

    public function enqueue_pwa_scripts() {
        // Register SW
        wp_add_inline_script('jquery', '
            if ("serviceWorker" in navigator) {
                navigator.serviceWorker.register("' . esc_js(home_url('/blyxxa-sw.js')) . '")
                .then(function(reg) { console.log("Blyxxa Client PWA: SW Registered"); });
            }
            
            // Offline Detection
            window.addEventListener("offline", function() {
                var div = document.createElement("div");
                div.id = "blyxxa-offline-alert";
                div.style.cssText = "position:fixed; bottom:80px; left:50%; transform:translateX(-50%); background:#ef4444; color:white; padding:10px 20px; border-radius:50px; z-index:99999; font-family:sans-serif; box-shadow:0 10px 25px rgba(0,0,0,0.2); font-weight:bold;";
                div.innerHTML = "⚠️ ' . esc_js(__('No Internet Connection. AI Tools may not work.', 'blyxxa-core')) . '";
                document.body.appendChild(div);
            });

            window.addEventListener("online", function() {
                var div = document.getElementById("blyxxa-offline-alert");
                if(div) {
                    div.style.background = "#10b981";
                    div.innerHTML = "✅ ' . esc_js(__('Back Online!', 'blyxxa-core')) . '";
                    setTimeout(function(){ div.remove(); }, 3000);
                }
            });

            // PWA Mode Detection
            if (window.matchMedia("(display-mode: standalone)").matches || window.navigator.standalone === true) {
                document.body.classList.add("blyxxa-pwa-mode");
            }
        ');

        // App Mode CSS
        wp_register_style('blyxxa-pwa-style', false, [], '1.0');
        wp_enqueue_style('blyxxa-pwa-style');
        
        // App Mode JS (Search & Logic)
        wp_register_script('blyxxa-pwa-js', false, [], '1.0', true);
        wp_enqueue_script('blyxxa-pwa-js');
        
        wp_add_inline_script('blyxxa-pwa-js', '
            document.addEventListener("DOMContentLoaded", function() {
                var searchInput = document.getElementById("blyxxa-app-search");
                if(searchInput){
                    searchInput.addEventListener("keyup", function() {
                        let val = this.value.toLowerCase();
                        document.querySelectorAll(".blyx-app-tool-card").forEach(el => {
                            el.style.display = el.getAttribute("data-search").includes(val) ? "flex" : "none";
                        });
                    });
                }
            });
        ');
        wp_add_inline_style('blyxxa-pwa-style', '
            /* Only apply in PWA Mode */
            body.blyxxa-pwa-mode header, 
            body.blyxxa-pwa-mode footer, 
            body.blyxxa-pwa-mode #masthead, 
            body.blyxxa-pwa-mode #colophon,
            body.blyxxa-pwa-mode .site-header,
            body.blyxxa-pwa-mode .site-footer {
                display: none !important;
            }
            
            body.blyxxa-pwa-mode {
                padding-bottom: 80px !important; /* Space for bottom bar */
                background: #f8fafc;
            }

            /* Bottom Navigation Bar */
            .blyxxa-bottom-nav {
                display: none;
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                background: #ffffff;
                border-top: 1px solid #e2e8f0;
                padding: 10px 0;
                padding-bottom: env(safe-area-inset-bottom);
                z-index: 99999;
                box-shadow: 0 -5px 20px rgba(0,0,0,0.05);
                justify-content: space-around;
                align-items: center;
            }

            body.blyxxa-pwa-mode .blyxxa-bottom-nav {
                display: flex;
            }

            .blyxxa-nav-item {
                text-decoration: none;
                color: #64748b;
                font-family: "Inter", sans-serif;
                font-size: 10px;
                text-align: center;
                display: flex;
                flex-direction: column;
                align-items: center;
                gap: 4px;
                width: 25%;
            }

            .blyxxa-nav-item.active {
                color: #06b6d4;
                font-weight: 600;
            }

            .blyxxa-nav-icon {
                font-size: 20px;
                margin-bottom: 2px;
            }
        ');
    }

    public function render_app_bar() {
        // Only render if user is logged in
        if (!is_user_logged_in()) return;

        $dashboard_url = admin_url('admin.php?page=blyxxa-hq');
        $active_tab = $_GET['tab'] ?? 'overview';
        
        // --- TOOL CATALOG LOGIC ---
        $args = [
            'post_type' => ['page', 'post'],
            'meta_key' => 'blyxxa_app_show',
            'meta_value' => '1',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
            'no_found_rows' => true // Optimization: Don't count total rows
        ];
        $tools_query = new WP_Query($args);
        $catalog = [];
        
        if ($tools_query->have_posts()) {
            while ($tools_query->have_posts()) {
                $tools_query->the_post();
                $cat = get_post_meta(get_the_ID(), 'blyxxa_app_category', true) ?: 'General';
                $icon = get_post_meta(get_the_ID(), 'blyxxa_app_icon', true) ?: '🛠️';
                $catalog[$cat][] = [
                    'title' => get_the_title(),
                    'url' => get_permalink(),
                    'icon' => $icon
                ];
            }
            wp_reset_postdata();
        }
        // --------------------------

        // Catalog Modal HTML
        echo '<div id="blyxxa-tool-catalog" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:#f8fafc; z-index:99998; padding:20px; padding-bottom:100px; overflow-y:auto;">
            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:20px;">
                <h2 style="margin:0; color:#0f172a;">' . esc_html__('Tools', 'blyxxa-core') . '</h2>
                <button onclick="document.getElementById(\'blyxxa-tool-catalog\').style.display=\'none\'" style="background:none; border:none; font-size:24px; color:#64748b;">&times;</button>
            </div>
            
            <input type="text" id="blyxxa-app-search" placeholder="' . esc_attr__('Search tools...', 'blyxxa-core') . '" style="width:100%; padding:12px; border:1px solid #cbd5e1; border-radius:10px; margin-bottom:20px; font-size:16px;">
            
            <div id="blyxxa-catalog-list">';
            
            if (empty($catalog)) {
                echo '<div style="text-align:center; color:#94a3b8; margin-top:50px;">' . esc_html__('No tools found. Add them via Page Settings.', 'blyxxa-core') . '</div>';
            } else {
                foreach ($catalog as $cat_name => $tools) {
                    echo '<h3 style="color:#64748b; font-size:14px; text-transform:uppercase; margin:20px 0 10px 0; border-bottom:1px solid #e2e8f0; padding-bottom:5px;">' . esc_html($cat_name) . '</h3>';
                    echo '<div style="display:grid; grid-template-columns: 1fr 1fr; gap:15px;">';
                    foreach ($tools as $tool) {
                        echo '<a href="' . esc_url($tool['url']) . '" class="blyx-app-tool-card" data-search="' . esc_attr(strtolower($tool['title'])) . '" style="background:#fff; padding:15px; border-radius:12px; text-decoration:none; color:#334155; box-shadow:0 2px 5px rgba(0,0,0,0.05); display:flex; flex-direction:column; align-items:center; text-align:center; gap:8px;">
                            <span style="font-size:24px;">' . esc_html($tool['icon']) . '</span>
                            <span style="font-weight:600; font-size:13px; line-height:1.3;">' . esc_html($tool['title']) . '</span>
                        </a>';
                    }
                    echo '</div>';
                }
            }
            
        echo '</div></div>';
        
        // Search Script moved to enqueue_pwa_scripts via wp_add_inline_script

        // Bottom Nav
        echo '<div class="blyxxa-bottom-nav">';
        
        // Home
        echo '<a href="' . esc_url(add_query_arg('tab', 'overview', $dashboard_url)) . '" class="blyxxa-nav-item ' . ($active_tab === 'overview' ? 'active' : '') . '">';
        echo '<span class="blyxxa-nav-icon">📊</span>';
        echo '<span>' . esc_html__('Overview', 'blyxxa-core') . '</span>';
        echo '</a>';

        // News
        echo '<a href="' . esc_url(add_query_arg('tab', 'announcements', $dashboard_url)) . '" class="blyxxa-nav-item ' . ($active_tab === 'announcements' ? 'active' : '') . '">';
        echo '<span class="blyxxa-nav-icon">📢</span>';
        echo '<span>' . esc_html__('News', 'blyxxa-core') . '</span>';
        echo '</a>';

        // Tools (Trigger Modal)
        echo '<a href="javascript:void(0)" onclick="document.getElementById(\'blyxxa-tool-catalog\').style.display=\'block\'" class="blyxxa-nav-item">';
        echo '<span class="blyxxa-nav-icon">🛠️</span>';
        echo '<span>' . esc_html__('Tools', 'blyxxa-core') . '</span>';
        echo '</a>';

        // Settings
        echo '<a href="' . esc_url(add_query_arg('tab', 'config', $dashboard_url)) . '" class="blyxxa-nav-item ' . ($active_tab === 'config' ? 'active' : '') . '">';
        echo '<span class="blyxxa-nav-icon">⚙️</span>';
        echo '<span>' . esc_html__('Settings', 'blyxxa-core') . '</span>';
        echo '</a>';

        echo '</div>';
    }

    private function render_service_worker() {
        header('Content-Type: application/javascript');
        $offline_msg = esc_html__('You are offline. Please check your connection.', 'blyxxa-core');
        
        echo "
        const CACHE_NAME = 'blyxxa-client-pwa-v1';
        const OFFLINE_URL = '" . esc_js(home_url()) . "';

        self.addEventListener('install', (event) => {
            self.skipWaiting();
        });

        self.addEventListener('activate', (event) => {
            event.waitUntil(clients.claim());
        });

        self.addEventListener('fetch', (event) => {
            // Network First strategy for API and HTML
            if (event.request.mode === 'navigate' || event.request.url.includes('/wp-json/')) {
                event.respondWith(
                    fetch(event.request).catch(() => {
                        return new Response('<h1>" . esc_js($offline_msg) . "</h1>', {
                            headers: { 'Content-Type': 'text/html' }
                        });
                    })
                );
            } else {
                // Cache First for assets
                event.respondWith(
                    caches.match(event.request).then((response) => {
                        return response || fetch(event.request);
                    })
                );
            }
        });
        ";
        exit;
    }

    private function render_manifest() {
        $settings = get_option('blyxxa_client_pwa_settings', []);
            
        $manifest = [
            'name' => $settings['name'] ?? get_bloginfo('name'),
            'short_name' => $settings['short_name'] ?? get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'start_url' => home_url('/'),
            'display' => 'standalone',
            'background_color' => $settings['bg_color'] ?? '#ffffff',
            'theme_color' => $settings['theme_color'] ?? '#0f172a',
            'icons' => []
        ];

        if (!empty($settings['icon'])) {
            $manifest['icons'][] = [
                'src' => $settings['icon'],
                'sizes' => '192x192',
                'type' => 'image/png'
            ];
            $manifest['icons'][] = [
                'src' => $settings['icon'],
                'sizes' => '512x512',
                'type' => 'image/png'
            ];
        }

        header('Content-Type: application/json');
        echo json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        exit;
    }

    public function output_meta_tags() {
        $settings = get_option('blyxxa_client_pwa_settings', []);
        $theme_color = $settings['theme_color'] ?? '#0f172a';
        
        echo '<meta name="theme-color" content="' . esc_attr($theme_color) . '">' . "\n";
        echo '<link rel="manifest" href="' . esc_url(home_url('/blyxxa-manifest.json')) . '">' . "\n";
        echo '<meta name="apple-mobile-web-app-capable" content="yes">' . "\n";
        echo '<meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">' . "\n";
        
        if (!empty($settings['icon'])) {
            echo '<link rel="apple-touch-icon" href="' . esc_url($settings['icon']) . '">' . "\n";
        }
    }
}

new Blyxxa_PWA_Client();
