<?php
// BLYXXA PWA FRONTEND
// Handles Manifest, Service Worker, and PWA UI Injection

// 1. DYNAMIC ENDPOINTS (Manifest & SW)
add_action('init', 'blyxxa_pwa_endpoints');
function blyxxa_pwa_endpoints() {
    // Rewrite Rule for .well-known/assetlinks.json
    add_rewrite_rule('^.well-known/assetlinks.json', 'index.php?blyxxa_pwa=assetlinks', 'top');

    if (isset($_GET['blyxxa_pwa'])) {
        $type = sanitize_text_field($_GET['blyxxa_pwa']);
        
        if ($type === 'manifest') {
            blyxxa_render_manifest();
        } elseif ($type === 'sw') {
            blyxxa_render_sw();
        } elseif ($type === 'assetlinks') {
            blyxxa_render_assetlinks();
        }
    }
}

// Render Manifest.json
function blyxxa_render_manifest() {
    header('Content-Type: application/json; charset=utf-8');
    
    $name = get_option('blyxxa_pwa_name', get_bloginfo('name'));
    $short_name = get_option('blyxxa_pwa_short_name', substr(get_bloginfo('name'), 0, 12));
    $desc = get_option('blyxxa_pwa_description', get_bloginfo('description'));
    $start_url = site_url('?blyxxa_pwa_home=1'); // Start at custom PWA home
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    $bg_color = get_option('blyxxa_pwa_bg_color', '#ffffff');
    $icon = get_option('blyxxa_pwa_icon', '');
    
    // Default Icon if missing (use a placeholder or site icon)
    if (empty($icon)) {
        $icon = get_site_icon_url(512); 
    }

    $manifest = [
        "name" => $name,
        "short_name" => $short_name,
        "description" => $desc,
        "start_url" => $start_url,
        "display" => "standalone",
        "background_color" => $bg_color,
        "theme_color" => $theme_color,
        "orientation" => "portrait",
        "icons" => []
    ];

    if ($icon) {
        $manifest["icons"][] = [
            "src" => $icon,
            "sizes" => "512x512",
            "type" => "image/png"
        ];
        $manifest["icons"][] = [
            "src" => $icon,
            "sizes" => "192x192",
            "type" => "image/png"
        ];
    }

    echo json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

// Render AssetLinks.json (Play Store TWA)
function blyxxa_render_assetlinks() {
    header('Content-Type: application/json; charset=utf-8');
    
    $package = get_option('blyxxa_pwa_assetlinks_package', '');
    $sha256_raw = get_option('blyxxa_pwa_assetlinks_sha256', '');
    
    if (empty($package) || empty($sha256_raw)) {
        echo json_encode([]); // Return empty array if not configured
        exit;
    }
    
    // Process SHA256 (split by newlines)
    $sha256_list = array_filter(array_map('trim', explode("\n", $sha256_raw)));
    
    $assetlinks = [[
        "relation" => ["delegate_permission/common.handle_all_urls"],
        "target" => [
            "namespace" => "android_app",
            "package_name" => $package,
            "sha256_cert_fingerprints" => array_values($sha256_list)
        ]
    ]];

    echo json_encode($assetlinks, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

// Render Service Worker (sw.js)
function blyxxa_render_sw() {
    header('Content-Type: application/javascript; charset=utf-8');
    header('Service-Worker-Allowed: /');
    
    ?>
    const CACHE_NAME = 'blyxxa-pwa-v4';
    const OFFLINE_URL = '<?php echo esc_js(site_url('?blyxxa_pwa_home=1')); ?>';
    
    const urlsToCache = [
        '/',
        OFFLINE_URL,
        '<?php echo esc_js(site_url('?blyxxa_pwa_tools=1')); ?>',
        '<?php echo esc_js(site_url('?blyxxa_pwa_profile=1')); ?>',
        '<?php echo esc_js(site_url('?blyxxa_pwa_history=1')); ?>',
        '<?php echo esc_js(includes_url('css/dashicons.min.css')); ?>'
    ];

    // 1. INSTALL: Cache App Shell
    self.addEventListener('install', event => {
        event.waitUntil(
            caches.open(CACHE_NAME)
                .then(cache => {
                    console.log('[SW] Caching App Shell');
                    return cache.addAll(urlsToCache);
                })
                .then(() => self.skipWaiting())
                .catch(err => console.error('[SW] Cache Failed:', err))
        );
    });

    // 2. ACTIVATE: Clean old caches
    self.addEventListener('activate', event => {
        event.waitUntil(
            caches.keys().then(keyList => {
                return Promise.all(keyList.map(key => {
                    if (key !== CACHE_NAME) {
                        console.log('[SW] Removing old cache', key);
                        return caches.delete(key);
                    }
                }));
            })
        );
        return self.clients.claim();
    });

    // 3. FETCH: Stale-While-Revalidate Strategy
    self.addEventListener('fetch', event => {
        // Skip non-GET requests and admin pages (except ajax)
        if (event.request.method !== 'GET' || event.request.url.includes('wp-admin')) {
            return;
        }

        event.respondWith(
            caches.match(event.request).then(cachedResponse => {
                const fetchPromise = fetch(event.request).then(networkResponse => {
                    // Update cache with new response
                    if (networkResponse && networkResponse.status === 200 && networkResponse.type === 'basic') {
                        const responseToCache = networkResponse.clone();
                        caches.open(CACHE_NAME).then(cache => {
                            cache.put(event.request, responseToCache);
                        });
                    }
                    return networkResponse;
                });

                // 1. Cache Hit: Return immediately, update in background
                if (cachedResponse) {
                    fetchPromise.catch(() => { /* Background fetch failed - ignore */ });
                    return cachedResponse;
                }

                // 2. Cache Miss: Wait for network
                return fetchPromise.catch(error => {
                    // 3. Network Failed & No Cache: Show Offline Page
                    if (event.request.mode === 'navigate') {
                        return caches.match(OFFLINE_URL);
                    }
                    throw error;
                });
            })
        );
    });
    <?php
    exit;
}

// 2. INJECT META TAGS (Head)
add_action('wp_head', 'blyxxa_pwa_meta_tags');
function blyxxa_pwa_meta_tags() {
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    $icon = get_option('blyxxa_pwa_icon', get_site_icon_url(192));
    
    echo '<!-- Blyxxa PWA Meta Tags -->';
    echo '<link rel="manifest" href="' . esc_url(site_url('?blyxxa_pwa=manifest')) . '">';
    echo '<meta name="theme-color" content="' . esc_attr($theme_color) . '">';
    echo '<meta name="apple-mobile-web-app-capable" content="yes">';
    echo '<meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">';
    echo '<meta name="apple-mobile-web-app-title" content="' . esc_attr(get_option('blyxxa_pwa_short_name')) . '">';
    
    if ($icon) {
        echo '<link rel="apple-touch-icon" href="' . esc_url($icon) . '">';
    }
}

// 3. INJECT PWA UI (Footer) - Install Banner & Bottom Nav & Offline UI
add_action('wp_footer', 'blyxxa_pwa_ui_elements');
function blyxxa_pwa_ui_elements() {
    if (is_admin()) return;

    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    
    // Register Service Worker
    ?>
    <script>
    if ('serviceWorker' in navigator) {
        window.addEventListener('load', function() {
            navigator.serviceWorker.register('<?php echo esc_js(site_url('?blyxxa_pwa=sw')); ?>', { scope: '/' })
            .then(reg => console.log('[SW] Registered'))
            .catch(err => console.log('[SW] Error', err));
        });
    }
    </script>
    
    <style>
        /* PWA Bottom Nav */
        #blyxxa-pwa-nav { display: none; }
        
        @media all and (display-mode: standalone) {
            header, footer, .site-header, .site-footer { display: none !important; }
            #blyxxa-pwa-nav {
                display: flex; position: fixed; bottom: 0; left: 0; width: 100%;
                background: #fff; border-top: 1px solid #e2e8f0;
                padding-bottom: env(safe-area-inset-bottom);
                z-index: 99999; box-shadow: 0 -4px 6px rgba(0,0,0,0.05);
            }
            #blyxxa-pwa-nav a {
                flex: 1; text-align: center; padding: 12px 0;
                color: #64748b; text-decoration: none; font-size: 10px;
                display: flex; flex-direction: column; align-items: center; gap: 4px;
            }
            #blyxxa-pwa-nav a.active { color: <?php echo esc_attr($theme_color); ?>; font-weight: 700; }
            #blyxxa-pwa-nav svg { width: 24px; height: 24px; fill: currentColor; }
            body { padding-bottom: 80px; padding-top: 20px; }
        }
        
        /* Install Banner (Mobile Browser Only) */
        #blyxxa-install-banner {
            display: none; position: fixed; top: 0; left: 0; width: 100%;
            background: <?php echo esc_attr($theme_color); ?>; color: #fff; z-index: 100000;
            padding: 10px 15px; box-shadow: 0 2px 10px rgba(0,0,0,0.2);
            align-items: center; justify-content: space-between; box-sizing: border-box;
        }
        #blyxxa-install-banner .btn {
            background: #fff; color: <?php echo esc_attr($theme_color); ?>; border: none;
            padding: 5px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; cursor: pointer;
        }

        /* Offline Toast */
        #blyxxa-offline-toast {
            display: none; position: fixed; bottom: 90px; left: 50%; transform: translateX(-50%);
            background: #333; color: #fff; padding: 10px 20px; border-radius: 25px;
            font-size: 13px; z-index: 100001; box-shadow: 0 4px 10px rgba(0,0,0,0.3);
            align-items: center; gap: 10px;
        }
        /* iOS Install Modal */
        #blyxxa-ios-install-modal {
            display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.8); z-index: 100002;
            align-items: center; justify-content: center;
        }
        .blyxxa-ios-pwa-box {
            background: #fff; width: 80%; max-width: 320px; border-radius: 16px; padding: 25px;
            text-align: center; color: #334155; position: relative;
        }
        .blyxxa-ios-pwa-box h3 { margin-top: 0; color: #0f172a; }
        .blyxxa-ios-steps { text-align: left; margin: 20px 0; font-size: 14px; line-height: 1.6; }
        .blyxxa-ios-steps div { margin-bottom: 10px; display: flex; align-items: center; gap: 10px; }
        .blyxxa-ios-icon { font-size: 20px; }
    </style>

    <!-- Offline Toast -->
    <div id="blyxxa-offline-toast">
        <span>📡</span> <?php esc_html_e('You are offline. Using cached mode.', 'blyxxa-core'); ?>
    </div>

    <!-- IOS Install Modal -->
    <div id="blyxxa-ios-install-modal" onclick="this.style.display='none'">
        <div class="blyxxa-ios-pwa-box" onclick="event.stopPropagation()">
            <h3><?php esc_html_e('Install App', 'blyxxa-core'); ?></h3>
            <div class="blyxxa-ios-steps">
                <div><span class="blyxxa-ios-icon">1️⃣</span> <?php esc_html_e('Tap the Share button', 'blyxxa-core'); ?> <span style="font-size:16px;">⎋</span></div>
                <div><span class="blyxxa-ios-icon">2️⃣</span> <?php esc_html_e('Scroll down and tap', 'blyxxa-core'); ?> <br><strong><?php esc_html_e('Add to Home Screen', 'blyxxa-core'); ?></strong> <span>➕</span></div>
            </div>
            <button onclick="document.getElementById('blyxxa-ios-install-modal').style.display='none'" class="btn" style="background:#0f172a; color:#fff; width:100%; padding:10px; font-size:14px;"><?php esc_html_e('Close', 'blyxxa-core'); ?></button>
        </div>
    </div>

    <!-- Install Banner -->
    <div id="blyxxa-install-banner">
        <div style="display:flex; align-items:center; gap:10px;">
            <span style="font-size:20px;">📱</span>
            <div style="font-size:13px; font-weight:600;"><?php esc_html_e('Install App for better experience', 'blyxxa-core'); ?></div>
        </div>
        <div style="display:flex; gap:10px; align-items:center;">
            <button class="btn" id="blyxxa-install-btn"><?php esc_html_e('Install', 'blyxxa-core'); ?></button>
            <button onclick="document.getElementById('blyxxa-install-banner').style.display='none'" style="background:none; border:none; color:#fff; font-size:18px; cursor:pointer;">&times;</button>
        </div>
    </div>

    <!-- Bottom Navigation Bar -->
    <div id="blyxxa-pwa-nav">
        <a href="<?php echo esc_url(site_url('?blyxxa_pwa_home=1')); ?>" class="<?php echo isset($_GET['blyxxa_pwa_home']) ? 'active' : ''; ?>">
            <svg viewBox="0 0 24 24"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>
            <span><?php esc_html_e('Home', 'blyxxa-core'); ?></span>
        </a>
        <a href="<?php echo esc_url(site_url('?blyxxa_pwa_tools=1')); ?>" class="<?php echo isset($_GET['blyxxa_pwa_tools']) ? 'active' : ''; ?>">
            <svg viewBox="0 0 24 24"><path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/></svg>
            <span><?php esc_html_e('Tools', 'blyxxa-core'); ?></span>
        </a>
        <a href="<?php echo esc_url(site_url('?blyxxa_pwa_history=1')); ?>" class="<?php echo isset($_GET['blyxxa_pwa_history']) ? 'active' : ''; ?>">
            <svg viewBox="0 0 24 24"><path d="M13 3c-4.97 0-9 4.03-9 9H1l3.89 3.89.07.14L9 12H6c0-3.87 3.13-7 7-7s7 3.13 7 7-3.13 7-7 7c-1.93 0-3.68-.79-4.94-2.06l-1.42 1.42C8.27 19.99 10.51 21 13 21c4.97 0 9-4.03 9-9s-4.03-9-9-9zm-1 5v5l4.28 2.54.72-1.21-3.5-2.08V8H12z"/></svg>
            <span><?php esc_html_e('History', 'blyxxa-core'); ?></span>
        </a>
        <a href="<?php echo esc_url(site_url('?blyxxa_pwa_profile=1')); ?>" class="<?php echo isset($_GET['blyxxa_pwa_profile']) ? 'active' : ''; ?>">
            <svg viewBox="0 0 24 24"><path d="M19.14 12.94c.04-.3.06-.61.06-.94 0-.32-.02-.64-.07-.94l2.03-1.58c.18-.14.23-.41.12-.61l-1.92-3.32c-.12-.22-.37-.29-.59-.22l-2.39.96c-.5-.38-1.03-.7-1.62-.94l-.36-2.54c-.04-.24-.24-.41-.48-.41h-3.84c-.24 0-.43.17-.47.41l-.36 2.54c-.59.24-1.13.57-1.62.94l-2.39-.96c-.22-.08-.47 0-.59.22L2.74 8.87c-.12.21-.08.47.12.61l2.03 1.58c-.05.3-.09.63-.09.94s.02.64.07.94l-2.03 1.58c-.18.14-.23.41-.12.61l1.92 3.32c.12.22.37.29.59.22l2.39-.96c.5.38 1.03.7 1.62.94l.36 2.54c.05.24.24.41.48.41h3.84c.24 0 .44-.17.47-.41l.36-2.54c.59-.24 1.13-.56 1.62-.94l2.39.96c.22.08.47 0 .59-.22l1.92-3.32c.12-.22.07-.47-.12-.61l-2.01-1.58zM12 15.6c-1.98 0-3.6-1.62-3.6-3.6s1.62-3.6 3.6-3.6 3.6 1.62 3.6 3.6-1.62 3.6-3.6 3.6z"/></svg>
            <span><?php esc_html_e('Settings', 'blyxxa-core'); ?></span>
        </a>
    </div>

    <script>
    // Offline Detection
    function updateOnlineStatus() {
        const toast = document.getElementById('blyxxa-offline-toast');
        if (!navigator.onLine) {
            toast.style.display = 'flex';
            setTimeout(() => { toast.style.display = 'none'; }, 5000);
        } else {
            toast.style.display = 'none';
        }
    }
    window.addEventListener('online', updateOnlineStatus);
    window.addEventListener('offline', updateOnlineStatus);
    updateOnlineStatus(); // Check on load

    // Install Logic
    let deferredPrompt;
    window.addEventListener('beforeinstallprompt', (e) => {
        e.preventDefault();
        deferredPrompt = e;
        // Show banner if not standalone (Desktop & Mobile)
        const isStandalone = window.matchMedia('(display-mode: standalone)').matches;
        if (!isStandalone) {
            document.getElementById('blyxxa-install-banner').style.display = 'flex';
        }
    });

    document.getElementById('blyxxa-install-btn').addEventListener('click', (e) => {
        document.getElementById('blyxxa-install-banner').style.display = 'none';
        if (deferredPrompt) {
            deferredPrompt.prompt();
            deferredPrompt.userChoice.then((choiceResult) => {
                deferredPrompt = null;
            });
        } else {
            // Fallback for iOS or if prompt not available
            document.getElementById('blyxxa-ios-install-modal').style.display = 'flex';
        }
    });
    </script>
    <?php
}

// 4. VIRTUAL PAGES (Home, Tools, Profile)
add_filter('template_include', 'blyxxa_pwa_templates');
function blyxxa_pwa_templates($template) {
    if (isset($_GET['blyxxa_pwa_home'])) {
        blyxxa_render_pwa_home();
        exit;
    }
    if (isset($_GET['blyxxa_pwa_tools'])) {
        blyxxa_render_pwa_tools();
        exit;
    }
    if (isset($_GET['blyxxa_pwa_profile'])) {
        blyxxa_render_pwa_profile();
        exit;
    }
    if (isset($_GET['blyxxa_pwa_history'])) {
        blyxxa_render_pwa_history();
        exit;
    }
    if (isset($_GET['blyxxa_pwa_tool'])) {
        blyxxa_render_pwa_single_tool();
        exit;
    }
    return $template;
}

// --- PWA HOME ---
function blyxxa_render_pwa_home() {
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <?php wp_head(); ?>
        <style>
            body { font-family: 'Inter', sans-serif; background: #f8fafc; margin: 0; padding: 20px; padding-bottom: 100px; }
            .pwa-hero { text-align: center; padding: 40px 20px; background: #fff; border-radius: 16px; box-shadow: 0 4px 6px rgba(0,0,0,0.05); margin-bottom: 20px; }
            .pwa-hero h1 { color: <?php echo esc_attr($theme_color); ?>; margin: 0 0 10px 0; font-size: 24px; }
            .pwa-hero p { color: #64748b; margin: 0; font-size: 14px; }
            .pwa-btn { display: block; width: 100%; padding: 15px; background: <?php echo esc_attr($theme_color); ?>; color: #fff; text-align: center; border-radius: 8px; text-decoration: none; font-weight: bold; margin-bottom: 10px; box-sizing: border-box; }
            .pwa-btn.secondary { background: #fff; color: <?php echo esc_attr($theme_color); ?>; border: 1px solid <?php echo esc_attr($theme_color); ?>; }
        </style>
    </head>
    <body>
        <div class="pwa-hero">
            <h1><?php echo esc_html(get_option('blyxxa_pwa_name', get_bloginfo('name'))); ?></h1>
            <p><?php echo esc_html(get_option('blyxxa_pwa_description', get_bloginfo('description'))); ?></p>
        </div>

        <?php if (!is_user_logged_in()): ?>
            <div style="background:#fff; padding:20px; border-radius:12px; box-shadow:0 2px 4px rgba(0,0,0,0.02);">
                <h3 style="text-align:center; margin-top:0;"><?php esc_html_e('Welcome', 'blyxxa-core'); ?></h3>
                <?php echo do_shortcode('[blyxxa_login]'); ?>
                <div style="text-align:center; margin-top:20px; font-size:13px;">
                    <a href="<?php echo esc_url(site_url('?blyxxa_pwa_home=1&action=register')); ?>"><?php esc_html_e('Create an Account', 'blyxxa-core'); ?></a>
                </div>
                <?php if(isset($_GET['action']) && $_GET['action'] == 'register'): ?>
                    <hr>
                    <?php echo do_shortcode('[blyxxa_register]'); ?>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div style="display:grid; grid-template-columns: 1fr 1fr; gap:15px;">
                <a href="<?php echo esc_url(site_url('?blyxxa_pwa_tools=1')); ?>" class="pwa-btn">
                    <span style="font-size:24px; display:block; margin-bottom:5px;">⚡</span>
                    <?php esc_html_e('Tools', 'blyxxa-core'); ?>
                </a>
                <a href="<?php echo esc_url(site_url('?blyxxa_pwa_profile=1')); ?>" class="pwa-btn secondary">
                    <span style="font-size:24px; display:block; margin-bottom:5px;">⚙️</span>
                    <?php esc_html_e('Settings', 'blyxxa-core'); ?>
                </a>
            </div>
            
            <!-- Quick History -->
            <div style="margin-top:20px;">
                <h3 style="font-size:16px; margin-bottom:10px;"><?php esc_html_e('Recent Activity', 'blyxxa-core'); ?></h3>
                <?php 
                // Simple history list
                $uid = get_current_user_id();
                $history = get_user_meta($uid, 'blyxxa_history', true);
                if (!empty($history) && is_array($history)) {
                    $history = array_slice($history, 0, 3); // Already sorted newest first usually
                    foreach ($history as $i => $item) {
                        $uniq = 'home_hist_' . $i;
                        echo '<div style="background:#fff; border-radius:8px; margin-bottom:8px; border:1px solid #eee; overflow:hidden;">';
                        
                        // Header (Clickable)
                        echo '<div onclick="document.getElementById(\'' . esc_js($uniq) . '\').style.display=(document.getElementById(\'' . esc_js($uniq) . '\').style.display==\'none\')?\'block\':\'none\'" style="padding:10px; cursor:pointer; display:flex; justify-content:space-between; align-items:center;">';
                        echo '<div>';
                        echo '<strong style="font-size:13px; display:block;">' . esc_html($item['title'] ?? 'Untitled') . '</strong>';
                        echo '<span style="color:#999; font-size:11px;">' . esc_html($item['date'] ?? '') . '</span>';
                        echo '</div>';
                        echo '<span style="color:#ccc; font-size:10px;">▼</span>';
                        echo '</div>';
                        
                        // Content (Hidden)
                        echo '<div id="' . esc_attr($uniq) . '" style="display:none; padding:15px; border-top:1px solid #f0f0f0; background:#fafafa; font-size:13px; color:#333;">';
                        echo wp_kses_post($item['content'] ?? '');
                        echo '</div>';
                        
                        echo '</div>';
                    }
                    echo '<div style="text-align:center; margin-top:10px;"><a href="' . esc_url(site_url('?blyxxa_pwa_history=1')) . '" style="font-size:12px; color:#64748b; text-decoration:none;">' . esc_html__('View All History', 'blyxxa-core') . ' →</a></div>';
                } else {
                    echo '<p style="color:#999; font-size:13px;">' . esc_html__('No activity yet.', 'blyxxa-core') . '</p>';
                }
                ?>
            </div>
        <?php endif; ?>

        <?php wp_footer(); ?>
    </body>
    <?php
}

// ... (Tools and Profile functions remain unchanged) ...

// --- PWA HISTORY ---
function blyxxa_render_pwa_history() {
    if (!is_user_logged_in()) {
        wp_safe_redirect(site_url('?blyxxa_pwa_home=1'));
        exit;
    }
    
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <?php wp_head(); ?>
        <style>
            body { font-family: 'Inter', sans-serif; background: #f8fafc; margin: 0; padding: 20px; padding-bottom: 100px; }
            .section-title { font-size: 14px; font-weight: 700; color: #94a3b8; text-transform: uppercase; margin-bottom: 10px; margin-top: 20px; }
            .pwa-card { background: #fff; border-radius: 12px; padding: 15px; box-shadow: 0 2px 4px rgba(0,0,0,0.02); margin-bottom: 15px; }
            
            /* Override plugin styles for mobile */
            .um-field { margin-bottom: 0 !important; }
        </style>
    </head>
    <body>
        <h2 style="margin-top:0; margin-bottom:20px;"><?php esc_html_e('History', 'blyxxa-core'); ?></h2>
        
        <div class="pwa-card">
            <?php 
            if (function_exists('blyxxa_inject_history_dashboard')) {
                echo wp_kses_post(blyxxa_inject_history_dashboard(true, 50)); 
            }
            ?>
        </div>

        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
}

// --- PWA TOOLS ---
function blyxxa_render_pwa_tools() {
    if (!is_user_logged_in()) {
        wp_safe_redirect(site_url('?blyxxa_pwa_home=1'));
        exit;
    }
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <?php wp_head(); ?>
        <style>
            body { font-family: 'Inter', sans-serif; background: #f8fafc; margin: 0; padding: 20px; padding-bottom: 100px; }
            .tool-card { background: #fff; padding: 15px; border-radius: 12px; box-shadow: 0 2px 4px rgba(0,0,0,0.02); display: flex; align-items: center; gap: 15px; text-decoration: none; color: #334155; margin-bottom: 10px; border: 1px solid #e2e8f0; }
            .tool-icon { width: 40px; height: 40px; background: #f1f5f9; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 20px; flex-shrink: 0; overflow: hidden; }
            .tool-icon img { width: 100%; height: 100%; object-fit: cover; }
            .tool-title { font-weight: 600; font-size: 14px; }
            
            /* Accordion Styles */
            .accordion-item { background: #fff; border-radius: 12px; margin-bottom: 10px; overflow: hidden; border: 1px solid #e2e8f0; }
            .accordion-header { padding: 15px; font-weight: bold; cursor: pointer; display: flex; justify-content: space-between; align-items: center; background: #f8fafc; }
            .accordion-header::after { content: '▼'; font-size: 12px; color: #94a3b8; transition: transform 0.2s; }
            .accordion-item.active .accordion-header::after { transform: rotate(180deg); }
            .accordion-content { display: none; padding: 15px; }
            .accordion-item.active .accordion-content { display: block; }
        </style>
    </head>
    <body>
        <h2 style="margin-top:0; margin-bottom:20px;"><?php esc_html_e('Tools', 'blyxxa-core'); ?></h2>
        
        <?php
        // Get Tools
        $tools = glob(BLYXXA_TOOLS_PATH . '*.php');
        $tools_by_cat = [];
        
        // Excluded System Pages
        $excluded_pages = ['blyxxa-login', 'blyxxa-register', 'blyxxa-dashboard', 'blyxxa-profile', 'blyxxa-history'];
        
        // Default Icon
        $default_icon_url = get_option('blyxxa_pwa_default_tool_icon');

        foreach ($tools as $file) {
            $data = get_file_data($file, ['Category' => 'Category', 'Name' => 'Name']);
            $slug = basename($file, '.php');
            
            if (in_array($slug, $excluded_pages)) continue;
            
            $cat = !empty($data['Category']) ? $data['Category'] : __('General', 'blyxxa-core');
            
            // Fix Title: Replace underscores/hyphens with spaces and capitalize
            if (!empty($data['Name'])) {
                $name = $data['Name'];
            } else {
                $name = ucwords(str_replace(['_', '-'], ' ', $slug));
            }
            
            // Icon Logic: Featured Image -> Custom Default -> Lightning
            $icon_html = '⚡';
            $page_id = get_option("blyxxa_page_id_$slug");
            
            if ($page_id && has_post_thumbnail($page_id)) {
                $icon_html = get_the_post_thumbnail($page_id, 'thumbnail', ['style' => 'width:100%; height:100%; object-fit:cover;']);
            } elseif (!empty($default_icon_url)) {
                $icon_html = '<img src="' . esc_url($default_icon_url) . '" alt="icon">';
            }
            
            $tools_by_cat[$cat][] = [
                'title' => $name,
                'url' => site_url('?blyxxa_pwa_tool=' . $slug), // Use PWA virtual URL
                'icon' => $icon_html
            ];
        }
        
        if (empty($tools_by_cat)) {
            echo '<p>' . esc_html__('No tools found.', 'blyxxa-core') . '</p>';
        } else {
            foreach ($tools_by_cat as $cat => $tools) {
                echo '<div class="accordion-item">';
                echo '<div class="accordion-header" onclick="this.parentElement.classList.toggle(\'active\')">' . esc_html($cat) . '</div>';
                echo '<div class="accordion-content">';
                foreach ($tools as $tool) {
                    echo '<a href="' . esc_url($tool['url']) . '" class="tool-card">';
                    echo '<div class="tool-icon">' . wp_kses_post($tool['icon']) . '</div>';
                    echo '<div class="tool-title">' . esc_html($tool['title']) . '</div>';
                    echo '</a>';
                }
                echo '</div>';
                echo '</div>';
            }
        }
        ?>
        
        <script>
            // Open first accordion by default
            document.querySelector('.accordion-item')?.classList.add('active');
        </script>

        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
}

// --- PWA SINGLE TOOL RENDERER ---
function blyxxa_render_pwa_single_tool() {
    if (!is_user_logged_in()) {
        wp_safe_redirect(site_url('?blyxxa_pwa_home=1'));
        exit;
    }

    $slug = sanitize_title($_GET['blyxxa_pwa_tool']);
    $file = BLYXXA_TOOLS_PATH . $slug . '.php';

    if (!file_exists($file)) {
        wp_die(esc_html__('Tool not found.', 'blyxxa-core'));
    }

    // Include tool to register shortcode
    include_once $file;
    
    // Get Tool Data for Header
    $data = get_file_data($file, ['Name' => 'Name']);
    $title = !empty($data['Name']) ? $data['Name'] : ucwords(str_replace(['_', '-'], ' ', $slug));
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <?php wp_head(); ?>
        <style>
            body { font-family: 'Inter', sans-serif; background: #f8fafc; margin: 0; padding: 20px; padding-bottom: 100px; }
            .pwa-back-btn { display: inline-flex; align-items: center; gap: 5px; text-decoration: none; color: #64748b; font-weight: 600; margin-bottom: 20px; font-size: 14px; }
            /* Adjust Tool Card for Mobile */
            .blyx-card { margin: 0 !important; padding: 20px !important; width: 100% !important; box-sizing: border-box; box-shadow: none !important; border: 1px solid #e2e8f0 !important; }
            .blyx-head h2 { font-size: 24px !important; }
            .blyx-head p { font-size: 14px !important; }
            .blyx-grid { gap: 15px !important; }
            .blyx-btn { font-size: 16px !important; padding: 15px !important; }
        </style>
    </head>
    <body>
        <a href="<?php echo esc_url(site_url('?blyxxa_pwa_tools=1')); ?>" class="pwa-back-btn">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 12H5M12 19l-7-7 7-7"/></svg>
            <?php esc_html_e('Back to Tools', 'blyxxa-core'); ?>
        </a>

        <?php echo do_shortcode("[tool_$slug]"); ?>

        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
}

// --- PWA PROFILE ---
function blyxxa_render_pwa_profile() {
    if (!is_user_logged_in()) {
        wp_safe_redirect(site_url('?blyxxa_pwa_home=1'));
        exit;
    }
    
    $current_user = wp_get_current_user();
    
    // Handle Language Switch
    if (isset($_POST['blyxxa_pwa_lang']) && check_admin_referer('blyx_pwa_lang_switch')) {
        $lang = sanitize_text_field(wp_unslash($_POST['blyxxa_pwa_lang']));
        update_user_meta($current_user->ID, 'blyxxa_user_lang', $lang);
        echo "<script>window.location.href = window.location.href;</script>"; // Reload
        exit;
    }
    
    $curr_lang = get_user_meta($current_user->ID, 'blyxxa_user_lang', true);
    if(!$curr_lang) $curr_lang = get_locale();
    
    $theme_color = get_option('blyxxa_pwa_theme_color', '#0f172a');
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <?php wp_head(); ?>
        <style>
            body { font-family: 'Inter', sans-serif; background: #f8fafc; margin: 0; padding: 20px; padding-bottom: 100px; }
            .profile-header { display: flex; align-items: center; gap: 15px; margin-bottom: 30px; background: #fff; padding: 20px; border-radius: 12px; box-shadow: 0 2px 4px rgba(0,0,0,0.02); }
            .profile-avatar img { border-radius: 50%; width: 60px; height: 60px; }
            .profile-info h2 { margin: 0; font-size: 18px; color: #334155; }
            .profile-info p { margin: 0; color: #64748b; font-size: 13px; }
            
            .section-title { font-size: 14px; font-weight: 700; color: #94a3b8; text-transform: uppercase; margin-bottom: 10px; margin-top: 20px; }
            .pwa-card { background: #fff; border-radius: 12px; padding: 15px; box-shadow: 0 2px 4px rgba(0,0,0,0.02); margin-bottom: 15px; }
            
            /* Override plugin styles for mobile */
            .blyxxa-api-card { box-shadow: none !important; border: none !important; padding: 0 !important; margin: 0 !important; }
            .um-field { margin-bottom: 0 !important; }
        </style>
    </head>
    <body>
        <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:20px;">
             <div class="profile-header" style="margin-bottom:0; padding:10px; box-shadow:none; background:none;">
                <div class="profile-avatar" style="width:50px; height:50px;"><?php echo get_avatar($current_user->ID, 50); ?></div>
                <div class="profile-info">
                    <h2 style="font-size:16px;"><?php echo esc_html($current_user->display_name); ?></h2>
                </div>
            </div>
            
            <!-- Language Switcher -->
            <form method="post" style="margin:0;">
                <?php wp_nonce_field('blyx_pwa_lang_switch'); ?>
                <select name="blyxxa_pwa_lang" onchange="this.form.submit()" style="padding:5px; border-radius:5px; border:1px solid #ddd; font-size:12px; background:#fff;">
                    <option value="en_US" <?php selected($curr_lang, 'en_US'); ?>>🇬🇧 EN</option>
                    <option value="tr_TR" <?php selected($curr_lang, 'tr_TR'); ?>>🇹🇷 TR</option>
                    <option value="de_DE" <?php selected($curr_lang, 'de_DE'); ?>>🇩🇪 DE</option>
                    <option value="fr_FR" <?php selected($curr_lang, 'fr_FR'); ?>>🇫🇷 FR</option>
                    <option value="es_ES" <?php selected($curr_lang, 'es_ES'); ?>>🇪🇸 ES</option>
                    <option value="ru_RU" <?php selected($curr_lang, 'ru_RU'); ?>>🇷🇺 RU</option>
                </select>
            </form>
        </div>

        <!-- 2. API SETTINGS (Bottom) -->
        <div class="section-title">🔑 <?php esc_html_e('Settings', 'blyxxa-core'); ?></div>
        <div class="pwa-card">
            <?php 
            if (function_exists('blyxxa_api_form_localized')) {
                blyxxa_api_form_localized();
            }
            ?>
        </div>

        <div style="text-align:center; margin-top:30px;">
            <a href="<?php echo esc_url(wp_logout_url(site_url('?blyxxa_pwa_home=1'))); ?>" style="color:#ef4444; text-decoration:none; font-size:14px; font-weight:600;"><?php esc_html_e('Log Out', 'blyxxa-core'); ?></a>
        </div>

        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
}
