<?php
/**
 * Blyxxa Native Panel (Multi-Page Edition)
 */

// Enqueue Scripts & Styles
add_action('wp_enqueue_scripts', 'blyxxa_enqueue_premium_assets');
function blyxxa_enqueue_premium_assets() {
    wp_register_style('blyxxa-premium', plugins_url('../assets/css/blyxxa-premium.css', __FILE__), [], '1.1.0');
    
    // Inject Brand Color
    $color = get_option('blyxxa_brand_color', '#3ba1da');
    $custom_css = ":root { --blyx-primary: {$color}; }";
    wp_add_inline_style('blyxxa-premium', $custom_css);
}

// HELPER: Get Page URL by Title Slug
function blyxxa_get_page_url($slug_key) {
    $page_id = get_option("blyxxa_page_id_$slug_key");
    if($page_id) return get_permalink($page_id);
    return home_url(); // Fallback
}

// =================================================================
// 1. LOGIN SHORTCODE: [blyxxa_login]
// =================================================================
add_shortcode('blyxxa_login', 'blyxxa_login_shortcode');
function blyxxa_login_shortcode($atts) {
    if (is_user_logged_in()) {
        return "<script>window.location.href='".blyxxa_get_page_url('blyxxa-dashboard')."';</script>";
    }

    wp_enqueue_style('blyxxa-premium');
    $brand_logo = get_option('blyxxa_brand_logo');
    $brand_name = get_option('blyxxa_brand_name', 'Blyxxa Labs');
    $brand_footer = get_option('blyxxa_brand_footer', '© 2024 Blyxxa Labs');
    $error = '';

    if (isset($_POST['blyxxa_login']) && wp_verify_nonce($_POST['blyx_login_nonce'], 'blyxxa_login_action')) {
        $creds = array(
            'user_login'    => isset($_POST['log']) ? sanitize_text_field(wp_unslash($_POST['log'])) : '',
            'user_password' => isset($_POST['pwd']) ? wp_unslash($_POST['pwd']) : '',
            'remember'      => true
        );
        $user = wp_signon($creds, false);
        if (is_wp_error($user)) {
            $error = $user->get_error_message();
        } else {
            wp_safe_redirect(blyxxa_get_page_url('blyxxa-dashboard'));
            exit;
        }
    }

    ob_start();
    ?>
    <div class="blyx-container blyx-auth-wrapper">
        <div class="blyx-auth-card">
            <?php if($brand_logo): ?>
                <img src="<?php echo esc_url($brand_logo); ?>" alt="<?php echo esc_attr($brand_name); ?>" class="blyx-brand-logo">
            <?php else: ?>
                <h2 style="color:var(--blyx-primary); margin-bottom:20px;"><?php echo esc_html($brand_name); ?></h2>
            <?php endif; ?>

            <?php if($error): ?>
                <div style="background:#fee2e2; color:#dc2626; padding:10px; border-radius:6px; margin-bottom:20px; font-size:13px;">
                    <?php echo esc_html($error); ?>
                </div>
            <?php endif; ?>

            <div class="blyx-auth-title"><?php esc_html_e('Welcome Back', 'blyxxa-core'); ?></div>
            <div class="blyx-auth-subtitle"><?php esc_html_e('Please sign in to continue', 'blyxxa-core'); ?></div>
            <form method="post">
                <div class="blyx-form-group">
                    <label class="blyx-label"><?php esc_html_e('Username or Email', 'blyxxa-core'); ?></label>
                    <input type="text" name="log" class="blyx-input" required>
                </div>
                <div class="blyx-form-group">
                    <label class="blyx-label"><?php esc_html_e('Password', 'blyxxa-core'); ?></label>
                    <input type="password" name="pwd" class="blyx-input" required>
                </div>
                <?php wp_nonce_field('blyxxa_login_action', 'blyx_login_nonce'); ?>
                <button type="submit" name="blyxxa_login" class="blyx-btn"><?php esc_html_e('Sign In', 'blyxxa-core'); ?></button>
            </form>
            <div class="blyx-auth-footer">
                <?php esc_html_e('New here?', 'blyxxa-core'); ?> <a href="<?php echo esc_url(blyxxa_get_page_url('blyxxa-register')); ?>"><?php esc_html_e('Create Account', 'blyxxa-core'); ?></a>
            </div>
            
            <div style="margin-top:30px; font-size:11px; color:#cbd5e1;">
                <?php echo esc_html($brand_footer); ?>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

// =================================================================
// 2. REGISTER SHORTCODE: [blyxxa_register]
// =================================================================
add_shortcode('blyxxa_register', 'blyxxa_register_shortcode');
function blyxxa_register_shortcode($atts) {
    if (is_user_logged_in()) {
        return "<script>window.location.href='".blyxxa_get_page_url('blyxxa-dashboard')."';</script>";
    }

    wp_enqueue_style('blyxxa-premium');
    $brand_logo = get_option('blyxxa_brand_logo');
    $brand_name = get_option('blyxxa_brand_name', 'Blyxxa Labs');
    $brand_footer = get_option('blyxxa_brand_footer', '© 2024 Blyxxa Labs');
    $error = '';

    if (isset($_POST['blyxxa_register']) && wp_verify_nonce($_POST['blyx_reg_nonce'], 'blyxxa_register_action')) {
        $email = isset($_POST['user_email']) ? sanitize_email(wp_unslash($_POST['user_email'])) : '';
        $username = isset($_POST['user_login']) ? sanitize_user(wp_unslash($_POST['user_login'])) : '';
        
        if(username_exists($username) || email_exists($email)) {
            $error = __('Username or Email already exists.', 'blyxxa-core');
        } else {
            $password = wp_generate_password(12, false);
            $user_id = wp_create_user($username, $password, $email);
            if(!is_wp_error($user_id)) {
                wp_set_current_user($user_id);
                wp_set_auth_cookie($user_id);
                // translators: %s: brand name
                $subject = sprintf(__('Welcome to %s', 'blyxxa-core'), $brand_name);
                // translators: %s: user password
                $message = sprintf(__('Your password: %s', 'blyxxa-core'), $password);
                wp_mail($email, $subject, $message);
                wp_safe_redirect(blyxxa_get_page_url('blyxxa-dashboard'));
                exit;
            } else {
                $error = $user_id->get_error_message();
            }
        }
    }

    ob_start();
    ?>
    <div class="blyx-container blyx-auth-wrapper">
        <div class="blyx-auth-card">
            <?php if($brand_logo): ?>
                <img src="<?php echo esc_url($brand_logo); ?>" alt="<?php echo esc_attr($brand_name); ?>" class="blyx-brand-logo">
            <?php else: ?>
                <h2 style="color:var(--blyx-primary); margin-bottom:20px;"><?php echo esc_html($brand_name); ?></h2>
            <?php endif; ?>

            <?php if($error): ?>
                <div style="background:#fee2e2; color:#dc2626; padding:10px; border-radius:6px; margin-bottom:20px; font-size:13px;">
                    <?php echo esc_html($error); ?>
                </div>
            <?php endif; ?>

            <div class="blyx-auth-title"><?php esc_html_e('Create Account', 'blyxxa-core'); ?></div>
            <div class="blyx-auth-subtitle"><?php esc_html_e('Join us to access premium tools', 'blyxxa-core'); ?></div>
            <form method="post">
                <div class="blyx-form-group">
                    <label class="blyx-label"><?php esc_html_e('Username', 'blyxxa-core'); ?></label>
                    <input type="text" name="user_login" class="blyx-input" required>
                </div>
                <div class="blyx-form-group">
                    <label class="blyx-label"><?php esc_html_e('Email Address', 'blyxxa-core'); ?></label>
                    <input type="email" name="user_email" class="blyx-input" required>
                </div>
                <?php wp_nonce_field('blyxxa_register_action', 'blyx_reg_nonce'); ?>
                <button type="submit" name="blyxxa_register" class="blyx-btn"><?php esc_html_e('Sign Up', 'blyxxa-core'); ?></button>
            </form>
            <div class="blyx-auth-footer">
                <?php esc_html_e('Already have an account?', 'blyxxa-core'); ?> <a href="<?php echo esc_url(blyxxa_get_page_url('blyxxa-login')); ?>"><?php esc_html_e('Log In', 'blyxxa-core'); ?></a>
            </div>
            
            <div style="margin-top:30px; font-size:11px; color:#cbd5e1;">
                <?php echo esc_html($brand_footer); ?>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

// =================================================================
// 3. DASHBOARD WRAPPER (Common Layout)
// =================================================================
function blyxxa_render_dashboard_layout($active_tab, $content_callback) {
    if (!is_user_logged_in()) {
        return "<script>window.location.href='".blyxxa_get_page_url('blyxxa-login')."';</script>";
    }

    wp_enqueue_style('blyxxa-premium');
    $user = wp_get_current_user();
    $brand_name = get_option('blyxxa_brand_name', 'Blyxxa Labs');

    ob_start();
    ?>
    <div class="blyx-container">
        <div class="blyx-dashboard">
            <!-- Sidebar -->
            <div class="blyx-sidebar">
                <div style="margin-bottom:30px; padding:0 15px;">
                    <h3 style="margin:0; font-size:18px; font-weight:800; color:var(--blyx-primary);"><?php echo esc_html($brand_name); ?></h3>
                </div>
                <nav>
                    <a href="<?php echo esc_url(blyxxa_get_page_url('blyxxa-dashboard')); ?>" class="blyx-nav-item <?php echo $active_tab == 'dashboard' ? 'active' : ''; ?>">📊 <?php esc_html_e('Overview', 'blyxxa-core'); ?></a>
                    <a href="<?php echo esc_url(blyxxa_get_page_url('blyxxa-profile')); ?>" class="blyx-nav-item <?php echo $active_tab == 'profile' ? 'active' : ''; ?>">⚙️ <?php esc_html_e('Profile & Settings', 'blyxxa-core'); ?></a>
                    <a href="<?php echo esc_url(blyxxa_get_page_url('blyxxa-history')); ?>" class="blyx-nav-item <?php echo $active_tab == 'history' ? 'active' : ''; ?>">📂 <?php esc_html_e('History', 'blyxxa-core'); ?></a>
                    <a href="<?php echo esc_url(wp_logout_url(blyxxa_get_page_url('blyxxa-login'))); ?>" class="blyx-nav-item" style="color:#ef4444;">🚪 <?php esc_html_e('Logout', 'blyxxa-core'); ?></a>
                </nav>
            </div>

            <!-- Content -->
            <div class="blyx-content">
                <!-- Header -->
                <div class="blyx-header">
                    <h2 class="blyx-title"><?php echo esc_html(ucfirst($active_tab)); ?></h2>
                    <div class="blyx-user-info">
                        <div style="text-align:right;">
                            <div style="font-weight:600; font-size:14px;"><?php echo esc_html($user->display_name); ?></div>
                            <div style="font-size:12px; color:var(--blyx-text-muted);"><?php echo esc_html($user->user_email); ?></div>
                        </div>
                        <div class="blyx-avatar">
                            <?php echo esc_html(strtoupper(substr($user->display_name, 0, 1))); ?>
                        </div>
                    </div>
                </div>

                <!-- Main Content Area -->
                <div class="blyx-tab-content active" style="display:block;">
                    <?php call_user_func($content_callback, $user); ?>
                </div>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

// =================================================================
// 4. SHORTCODES FOR PAGES
// =================================================================

// [blyxxa_dashboard]
add_shortcode('blyxxa_dashboard', function() {
    return blyxxa_render_dashboard_layout('dashboard', function($user) {
        if(function_exists('blyxxa_status_localized')) {
            echo blyxxa_status_localized(true); 
        }
    });
});

// [blyxxa_profile]
add_shortcode('blyxxa_profile', function() {
    // Handle Language Save (PRG Pattern)
    if(isset($_POST['blyxxa_save_lang']) && is_user_logged_in() && wp_verify_nonce($_POST['blyx_lang_nonce'], 'save_lang')) {
        update_user_meta(get_current_user_id(), 'blyxxa_user_lang', isset($_POST['blyxxa_lang']) ? sanitize_text_field(wp_unslash($_POST['blyxxa_lang'])) : 'en_US');
        // Redirect to same page to avoid resubmission loop
        wp_safe_redirect(remove_query_arg('saved')); 
        exit;
    }

    return blyxxa_render_dashboard_layout('profile', function($user) {
        $current_lang = get_user_meta($user->ID, 'blyxxa_user_lang', true);
        if(!$current_lang) $current_lang = get_locale();
        ?>
        <div class="blyxxa-api-card">
            <h3 style="margin:0 0 15px 0;"><?php esc_html_e('Interface Language', 'blyxxa-core'); ?></h3>
            <form method="post">
                <select name="blyxxa_lang" class="blyx-input" style="max-width:300px;">
                    <option value="en_US" <?php selected($current_lang, 'en_US'); ?>>English (US)</option>
                    <option value="tr_TR" <?php selected($current_lang, 'tr_TR'); ?>>Türkçe</option>
                    <option value="ru_RU" <?php selected($current_lang, 'ru_RU'); ?>>Русский</option>
                    <option value="de_DE" <?php selected($current_lang, 'de_DE'); ?>>Deutsch</option>
                    <option value="fr_FR" <?php selected($current_lang, 'fr_FR'); ?>>Français</option>
                    <option value="es_ES" <?php selected($current_lang, 'es_ES'); ?>>Español</option>
                </select>
                <?php wp_nonce_field('save_lang', 'blyx_lang_nonce'); ?>
                <div style="margin-top:15px;">
                    <button type="submit" name="blyxxa_save_lang" class="blyx-btn-save"><?php esc_html_e('Save Changes', 'blyxxa-core'); ?></button>
                </div>
            </form>
        </div>
        <?php
        // API Keys
        if(function_exists('blyxxa_api_form_localized')) {
            echo blyxxa_api_form_localized(true); 
        }
    });
});

// [blyxxa_history]
add_shortcode('blyxxa_history', function() {
    return blyxxa_render_dashboard_layout('history', function($user) {
        if(function_exists('blyxxa_inject_history_dashboard')) {
            echo blyxxa_inject_history_dashboard(true); 
        }
    });
});
