<?php
if (!defined('ABSPATH')) exit;

class Blyxxa_License_Server {

    public function __construct() {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    public function register_routes() {
        // Add CORS Headers
        add_filter('rest_pre_serve_request', function($value) {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: POST, GET, OPTIONS, PUT, DELETE');
            header('Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization');
            return $value;
        });

        register_rest_route('blyxxa/v1', '/verify-license', [
            'methods' => 'POST',
            'callback' => [$this, 'verify_license'],
            'permission_callback' => '__return_true', // Public endpoint for clients
        ]);

        register_rest_route('blyxxa/v1', '/update-check', [
            'methods' => 'POST',
            'callback' => [$this, 'check_update'],
            'permission_callback' => '__return_true',
        ]);
    }

    public function verify_license($request) {
        $params = $request->get_params();
        $license_key = sanitize_text_field($params['license_key'] ?? '');
        $domain = sanitize_text_field($params['domain'] ?? '');
        $product_id = get_option('blyxxa_gumroad_plugin_product_id', ''); // Plugin Rental Product ID

        if (empty($license_key)) {
            return new WP_Error('missing_key', 'License key is required', ['status' => 400]);
        }

        // 1. Verify with Gumroad
        $response = wp_remote_post('https://api.gumroad.com/v2/licenses/verify', [
            'body' => [
                'product_id' => $product_id,
                'license_key' => $license_key,
                'increment_uses_count' => 'false' // Don't increment for checks
            ]
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('gumroad_error', $response->get_error_message(), ['status' => 500]);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['success']) && $body['success'] === true) {
            // 2. Log/Update Local License Tracking
            $this->log_license($license_key, $domain, $body['purchase']['email'] ?? 'unknown');

            // 3. Return Master Prompt & Subscription Data
            $master_prompt = get_option('blyxxa_master_prompt', '');
            
            // Fallback if empty (Safety Net for Master Node)
            if(empty($master_prompt)) {
                 $master_prompt = "Blyxxa Master Prompt (Fallback) - Please configure in Master Node settings.";
            }
            
            // Extract Subscription Info
            $sub_status = 'active';
            
            if (isset($body['purchase']['subscription_cancelled_at']) && $body['purchase']['subscription_cancelled_at']) {
                $sub_status = 'cancelled';
            }
            if (isset($body['purchase']['subscription_failed_at']) && $body['purchase']['subscription_failed_at']) {
                $sub_status = 'failed';
            }
            
            return rest_ensure_response([
                'success' => true,
                'master_prompt' => $master_prompt,
                'subscription' => [
                    'status' => $sub_status,
                    'email' => $body['purchase']['email'] ?? '',
                    'variant' => $body['purchase']['variants'] ?? ''
                ],
                'message' => 'License verified successfully.'
            ]);
        } else {
            return new WP_Error('invalid_license', 'Invalid License Key', ['status' => 403]);
        }
    }

    public function check_update($request) {
        // Logic to serve updates
        return rest_ensure_response([
            'version' => '1.0.0',
            'package' => '', // URL to zip file
            'requires' => '6.0',
            'tested' => '6.4'
        ]);
    }

    private function log_license($key, $domain, $email) {
        $licenses = get_option('blyxxa_client_licenses', []);
        
        // Update or Add
        $licenses[$key] = [
            'domain' => $domain,
            'email' => $email,
            'last_check' => current_time('mysql'),
            'status' => 'active'
        ];

        update_option('blyxxa_client_licenses', $licenses);
    }
}

new Blyxxa_License_Server();
