<?php
if (!defined('ABSPATH')) exit;

class Blyxxa_Client_License {

    private $api_url = BLYXXA_MASTER_API_URL;
    private $option_name = 'blyxxa_user_email'; // Changed from license_key
    private $cache_key = 'blyxxa_cached_master_prompt';
    private $sub_status_key = 'blyxxa_client_sub_status';

    public function __construct() {
        add_action('admin_init', [$this, 'handle_activation']);
        // Periodic Sync Hook (Twice Daily)
        add_action('blyxxa_daily_sync_event', [$this, 'sync_public_data']);
        
        // Tool Download AJAX
        add_action('wp_ajax_blyxxa_client_download_tool', [$this, 'handle_tool_download']);
    }

    public function handle_activation() {
        if (isset($_POST['blyxxa_activate_license']) && check_admin_referer('blyx_activate_license')) {
            $email = sanitize_text_field($_POST['blyxxa_user_email']); // Changed to text_field to support AppSumo codes
            $this->activate_license($email);
        }
        
        if (isset($_POST['blyxxa_deactivate_license']) && check_admin_referer('blyx_deactivate_license')) {
            delete_option($this->option_name);
            delete_option($this->sub_status_key);
            delete_transient($this->cache_key);
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p style="color:#333;">' . esc_html__('Subscription Deactivated.', 'blyxxa-core') . '</p></div>';
            });
        }

        // Handle Refresh - works for ALL users (licensed or not)
        if (isset($_POST['blyxxa_refresh_data']) && check_admin_referer('blyx_refresh_data')) {
            $this->sync_public_data();
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p style="color:#333;">✅ ' . esc_html__('Data refreshed!', 'blyxxa-core') . '</p></div>';
            });
        }
    }
    
    /**
     * Sync public data (changelog, broadcasts, branding) - no license required
     */
    public function sync_public_data() {
        $response = wp_remote_get($this->api_url . '/public-data', [
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($body['success']) && $body['success'] === true && !empty($body['data'])) {
            if (isset($body['data']['broadcasts'])) {
                update_option('blyxxa_client_broadcasts', $body['data']['broadcasts']);
            }
            if (isset($body['data']['changelog'])) {
                update_option('blyxxa_client_changelog', $body['data']['changelog']);
            }
            if (isset($body['data']['branding'])) {
                update_option('blyxxa_client_branding', $body['data']['branding']);
            }
            return true;
        }
        
        return false;
    }

    public function activate_license($email) {
        $domain = $_SERVER['HTTP_HOST'];
        
        // We send 'license_key' param because the server expects it, but we pass the email
        $response = wp_remote_post($this->api_url . '/verify-license', [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'license_key' => $email, 
                'domain' => $domain
            ]),
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Connection Error: ', 'blyxxa-core') . esc_html($response->get_error_message()) . '</p></div>';
            return;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        // ALWAYS SYNC PUBLIC DATA (Broadcasts, Changelog, Branding)
        // This ensures free users still get announcements and updates
        if (isset($body['broadcasts'])) update_option('blyxxa_client_broadcasts', $body['broadcasts']);
        if (isset($body['changelog'])) update_option('blyxxa_client_changelog', $body['changelog']);
        if (isset($body['branding'])) update_option('blyxxa_client_branding', $body['branding']);

        if (!empty($body['success']) && $body['success'] === true) {
            update_option($this->option_name, $email);
            
            // Cache Prompt (24 Hours) - ENCRYPTED
            if (!empty($body['master_prompt'])) {
                $encrypted_prompt = $this->encrypt_data($body['master_prompt'], $email);
                set_transient($this->cache_key, $encrypted_prompt, 24 * HOUR_IN_SECONDS);
            }
            
            // Save Backup & Timestamp for Grace Period
            update_option('blyxxa_backup_master_prompt', $encrypted_prompt);
            update_option('blyxxa_last_successful_check', time());
            
            // Save Subscription Status
            if (!empty($body['subscription'])) {
                update_option($this->sub_status_key, $body['subscription']);
            }
            
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p style="color:#333;">✅ ' . esc_html__('Subscription Verified Successfully!', 'blyxxa-core') . '</p></div>';
            });
        } else {
            // VERIFICATION FAILED
            // But we successfully synced public data, so we mention that.
            $msg = $body['message'] ?? __('Invalid Email or No Active Subscription Found.', 'blyxxa-core');
            add_action('admin_notices', function() use ($msg) {
                echo '<div class="notice notice-warning is-dismissible"><p style="color:#333;">⚠️ ' . esc_html__('Subscription Verification Failed: ', 'blyxxa-core') . esc_html($msg) . '</p><p style="font-size:12px; margin-top:5px; color:#64748b;"><em>' . esc_html__('However, public data (Broadcasts & Changelog) has been refreshed.', 'blyxxa-core') . '</em></p></div>';
            });
        }
    }

    public static function get_master_prompt() {
        $instance = new self();
        $email = get_option($instance->option_name);

        if (!$email) {
            return false; // No email
        }

        // 1. Check Cache
        $cached_encrypted = get_transient($instance->cache_key);
        if ($cached_encrypted) {
            $decrypted = $instance->decrypt_data($cached_encrypted, $email);
            if ($decrypted) {
                return $decrypted;
            }
        }

        // 2. Refresh from Server (Cache Miss, Expired, or Decryption Failed)
        $domain = $_SERVER['HTTP_HOST'];
        $response = wp_remote_post($instance->api_url . '/verify-license', [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'license_key' => $email,
                'domain' => $domain
            ]),
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            // GRACE PERIOD LOGIC (72 Hours)
            $last_check = get_option('blyxxa_last_successful_check', 0);
            if ((time() - $last_check) < (72 * HOUR_IN_SECONDS)) {
                // Within Grace Period - Try to use backup
                $backup_encrypted = get_option('blyxxa_backup_master_prompt');
                if ($backup_encrypted) {
                    $decrypted = $instance->decrypt_data($backup_encrypted, $email);
                    if ($decrypted) {
                        // Set a temporary flag for UI warning
                        set_transient('blyxxa_connection_warning', 'offline', 10 * MINUTE_IN_SECONDS);
                        return $decrypted;
                    }
                }
            }
            return false; 
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['success']) && $body['success'] === true && !empty($body['master_prompt'])) {
            // Encrypt before caching
            $encrypted_prompt = $instance->encrypt_data($body['master_prompt'], $email);
            set_transient($instance->cache_key, $encrypted_prompt, 24 * HOUR_IN_SECONDS);
            
            // Save Backup & Timestamp for Grace Period
            update_option('blyxxa_backup_master_prompt', $encrypted_prompt);
            update_option('blyxxa_last_successful_check', time());
            delete_transient('blyxxa_connection_warning'); // Clear warning if successful
            
            // Update Sub Status
            if (!empty($body['subscription'])) {
                update_option($instance->sub_status_key, $body['subscription']);
            }
            
            // Update Broadcasts (Array) & Changelog
            if (isset($body['broadcasts'])) update_option('blyxxa_client_broadcasts', $body['broadcasts']);
            if (isset($body['changelog'])) update_option('blyxxa_client_changelog', $body['changelog']);
            
            // Update Branding
            if (isset($body['branding'])) update_option('blyxxa_client_branding', $body['branding']);
            
            return $body['master_prompt'];
        }

        // License invalid or subscription cancelled - Clear Backup
        delete_option('blyxxa_backup_master_prompt');
        return false;
    }

    private function encrypt_data($data, $key) {
        $key = hash('sha256', $key, true); // Ensure 32-byte key
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length('aes-256-cbc'));
        $encrypted = openssl_encrypt($data, 'aes-256-cbc', $key, 0, $iv);
        return base64_encode($encrypted . '::' . $iv);
    }

    private function decrypt_data($data, $key) {
        $key = hash('sha256', $key, true); // Ensure 32-byte key
        $data = base64_decode($data);
        if (strpos($data, '::') === false) return false;
        list($encrypted_data, $iv) = explode('::', $data, 2);
        return openssl_decrypt($encrypted_data, 'aes-256-cbc', $key, 0, $iv);
    }
    
    public static function send_feedback($subject, $message) {
        $instance = new self();
        $key = get_option($instance->option_name);
        $sub = get_option($instance->sub_status_key, []);
        $email = $sub['email'] ?? get_option('admin_email');
        
        $response = wp_remote_post($instance->api_url . '/feedback', [
            'body' => json_encode([
                'license_key' => $key,
                'subject' => $subject,
                'message' => $message,
                'email' => $email,
                'user_type' => (self::get_license_status() === 'active') ? 'paid' : 'free'
            ]),
            'headers' => ['Content-Type' => 'application/json'],
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) return false;
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return !empty($body['success']);
    }
    
    public static function get_license_status() {
        $email = get_option('blyxxa_user_email');
        if(!$email) return 'unlicensed';
        
        $sub = get_option('blyxxa_client_sub_status', []);
        if(isset($sub['status']) && $sub['status'] !== 'active') return 'expired';
        
        return 'active';
    }

    /**
     * Handle Tool Download via AJAX
     */
    public function handle_tool_download() {
        check_ajax_referer('blyx_client_dl', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(__('Unauthorized', 'blyxxa-core'));
        
        $tool_id = sanitize_text_field($_POST['tool_id']);
        if (!$tool_id) wp_send_json_error(__('Missing tool ID', 'blyxxa-core'));
        
        $license_key = get_option('blyxxa_user_email'); // We use email as key for now
        $domain = $_SERVER['HTTP_HOST'];
        
        // 1. Call Master Node API
        $response = wp_remote_post($this->api_url . '/store/download', [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'tool_id' => $tool_id,
                'license_key' => $license_key,
                'domain' => $domain
            ]),
            'timeout' => 30 // Allow more time for code transfer
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error(__('Connection Error: ', 'blyxxa-core') . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($body['success']) && $body['success'] === true && !empty($body['content'])) {
            // 2. Validate & Save Tool
            $filename = sanitize_file_name($body['filename']);
            if (substr($filename, -4) !== '.php') $filename .= '.php'; // Ensure extension
            
            // Security: Ensure it starts with <?php
            $content = $body['content'];
            if (strpos(trim($content), '<?php') !== 0) $content = "<?php\n" . $content;
            
            if (!is_dir(BLYXXA_TOOLS_PATH)) wp_mkdir_p(BLYXXA_TOOLS_PATH);
            
            if (file_put_contents(BLYXXA_TOOLS_PATH . $filename, $content)) {
                // Clear Store Cache so UI updates button state
                delete_transient('blyxxa_store_data_v1');
                
                wp_send_json_success(['msg' => __('Tool installed successfully!', 'blyxxa-core')]);
            } else {
                wp_send_json_error(__('Failed to save tool file. Check folder permissions.', 'blyxxa-core'));
            }
        } else {
             $msg = $body['message'] ?? $body['data'] ?? __('Download failed.', 'blyxxa-core');
             wp_send_json_error($msg);
        }
    }
}

new Blyxxa_Client_License();
