<?php
/*
Plugin Name: Blyxxa Labs
Plugin URI: https://blyxxa.online
Description: The engine behind your AI-powered SaaS.
Version: 3.3.0
Author: Blyxxa LLC
Author URI: https://blyxxa.com
Text Domain: blyxxa-core
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// =================================================================
// 1. GLOBAL SYSTEM TWEAKS (TIMEOUT FIX)
// =================================================================
add_filter('http_request_timeout', function($timeout) { return 120; }, 9999);
add_action('http_api_curl', function($handle) {
    curl_setopt($handle, CURLOPT_CONNECTTIMEOUT, 30);
    curl_setopt($handle, CURLOPT_TIMEOUT, 120);
});
add_action('init', function() { set_time_limit(300); });

// =================================================================
// 2. CONSTANTS & LOADER
// =================================================================
define( 'BLYXXA_VERSION', '3.2.1' );
define( 'BLYXXA_CORE_PATH', plugin_dir_path( __FILE__ ) );
define( 'BLYXXA_CORE_URL', plugin_dir_url( __FILE__ ) );
define( 'BLYXXA_TOOLS_PATH', BLYXXA_CORE_PATH . 'tools/' );

// MASTER NODE API URL (UPDATE THIS BEFORE DISTRIBUTION)
define('BLYXXA_MASTER_API_URL', 'https://blyxxa.online/blyxxalabs/wp-json/blyxxa/v1');
define('BLYXXA_PLUGIN_SALES_URL', 'https://blyxxa.gumroad.com/l/BlyxxaSaasEngine'); // Where admins buy the plugin

// Include Engine
require_once BLYXXA_CORE_PATH . 'includes/blyxxa-engine.php';

// Load Text Domain
add_action( 'plugins_loaded', 'blyxxa_load_textdomain' );
function blyxxa_load_textdomain() {
    load_plugin_textdomain( 'blyxxa-core', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}

// Include Core Modules
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-api-handler.php';
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-engine.php';
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-ui.php';
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-native-panel.php'; // Premium Native Panel
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-updater.php'; // Auto Updater
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-pwa-frontend.php'; // PWA Frontend Logic (ADDED)
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-ping.php'; // Free User Tracking Ping

// Initialize Updater
new Blyxxa_Updater(__FILE__);

// Include Admin Modules
if ( is_admin() ) {
    require_once BLYXXA_CORE_PATH . 'admin/blyxxa-dashboard.php';
    require_once BLYXXA_CORE_PATH . 'admin/blyxxa-factory.php';
    // YENİ: Tool Manager modülü
    if ( file_exists( BLYXXA_CORE_PATH . 'admin/tool-manager.php' ) ) {
        require_once BLYXXA_CORE_PATH . 'admin/tool-manager.php';
    }
    if ( file_exists( BLYXXA_CORE_PATH . 'admin/blyxxa-pwa-manager.php' ) ) {
        require_once BLYXXA_CORE_PATH . 'admin/blyxxa-pwa-manager.php';
    }
    // Store (ADDED)
    if ( file_exists( BLYXXA_CORE_PATH . 'admin/blyxxa-store.php' ) ) {
        require_once BLYXXA_CORE_PATH . 'admin/blyxxa-store.php';
    }
}

// =================================================================
// 3. AUTO-LOADER (THE SHELF SYSTEM)
// =================================================================

// A. Eski "Kasa" Dosyasını Yükle (Geriye Dönük Uyumluluk)
if ( file_exists( BLYXXA_CORE_PATH . 'blyxxa-custom-tools.php' ) ) {
    require_once BLYXXA_CORE_PATH . 'blyxxa-custom-tools.php';
}

// B. Yeni "Tools" Klasöründeki Araçları Otomatik Yükle
// Bu döngü, tools klasörüne eklenen her .php dosyasını sisteme dahil eder.
$blyxxa_tools = glob( BLYXXA_TOOLS_PATH . '*.php' );
if ( ! empty( $blyxxa_tools ) ) {
    foreach ( $blyxxa_tools as $blyxxa_tool ) {
        require_once $blyxxa_tool;
    }
}

// =================================================================
// 4. ADMIN MENU STRUCTURE
// =================================================================

// Activation Hook
register_activation_hook(__FILE__, 'blyxxa_activate_plugin');
register_activation_hook(__FILE__, 'blyxxa_send_activation_ping'); // Send ping on activation

// Deactivation Hook - Clear scheduled events and send deactivation ping
register_deactivation_hook(__FILE__, 'blyxxa_send_deactivation_ping');
register_deactivation_hook(__FILE__, 'blyxxa_clear_weekly_ping_schedule');
register_deactivation_hook(__FILE__, 'blyxxa_clear_sync_schedule'); // Clear Sync Schedule

function blyxxa_clear_sync_schedule() {
    wp_clear_scheduled_hook('blyxxa_daily_sync_event');
}

function blyxxa_activate_plugin() {
    // Schedule Daily Sync (Broadcasts/Changelog)
    if (!wp_next_scheduled('blyxxa_daily_sync_event')) {
        wp_schedule_event(time(), 'twicedaily', 'blyxxa_daily_sync_event');
    }

    $pages = [
        __('Blyxxa Login', 'blyxxa-core')     => '[blyxxa_login]',
        __('Blyxxa Register', 'blyxxa-core')  => '[blyxxa_register]',
        __('Blyxxa Dashboard', 'blyxxa-core') => '[blyxxa_dashboard]',
        __('Blyxxa Profile', 'blyxxa-core')   => '[blyxxa_profile]',
        __('Blyxxa History', 'blyxxa-core')   => '[blyxxa_history]'
    ];

    foreach ($pages as $title => $content) {
        // $page_check = get_page_by_title($title); <- Deprecated
        $args = array(
            'title' => $title,
            'post_type' => 'page',
            'post_status' => 'publish',
            'posts_per_page' => 1
        );
        $query = new WP_Query($args);
        $page_check = $query->have_posts() ? $query->posts[0] : null;
        $page_id = 0;
        
        if ( $page_check && isset($page_check->ID) ) {
            $page_id = $page_check->ID;
        } else {
            $page_id = wp_insert_post([
                'post_title'   => $title,
                'post_content' => $content,
                'post_status'  => 'publish',
                'post_type'    => 'page',
                'post_author'  => 1,
            ]);
        }
        
        if ( $page_id && !is_wp_error($page_id) ) {
            $slug = sanitize_title($title);
            update_option("blyxxa_page_id_$slug", $page_id);
        }
    }
}

// User Language Filter
add_filter('locale', 'blyxxa_set_user_locale');
function blyxxa_set_user_locale($locale) {
    if (function_exists('is_user_logged_in') && is_user_logged_in()) {
        $user_lang = get_user_meta(get_current_user_id(), 'blyxxa_user_lang', true);
        if ($user_lang) {
            return $user_lang;
        }
    }
    return $locale;
}

// Enqueue Gumroad Script Globally
add_action('wp_enqueue_scripts', function() {
    wp_enqueue_script('gumroad-overlay', 'https://gumroad.com/js/gumroad.js', [], '1.0', true);
});

add_action('admin_menu', function() {
    // Main Menu: Blyxxa
    add_menu_page(
        __('Blyxxa Saas Engine', 'blyxxa-core'), 
        __('Blyxxa Saas Engine', 'blyxxa-core'), 
        'manage_options', 
        'blyxxa-hq', 
        'blyxxa_dashboard_page', 
        'dashicons-superhero', 
        2
    );



    // Sub Menu: Tool Factory
        add_submenu_page(
            'blyxxa-hq', // Parent slug
            'New Tool', // Page title
            'Tool Factory', // Menu title
            'manage_options', // Capability
            'blyxxa-tool-factory', // Menu slug
            'blyxxa_gen_page_v27' // Callback function
        ); // YENİ: Tool Manager Menüsü
    add_submenu_page(
        'blyxxa-hq',
        __('Tool Manager', 'blyxxa-core'),
        __('Tool Manager', 'blyxxa-core'),
        'manage_options',
        'blyxxa-tool-manager',
        'blyxxa_tool_manager_page'
    );

    // Custom Update Count
    $update_count = blyxxa_get_store_updates();
    $store_title = __('Store', 'blyxxa-core');
    if ($update_count > 0) {
        $store_title .= ' <span class="update-plugins count-' . intval($update_count) . '"><span class="plugin-count">' . intval($update_count) . '</span></span>';
    }

    // Sub Menu: Tool Store (MOVED & UPPERCASE)
    add_submenu_page(
        'blyxxa-hq',
        $store_title,
        'STORE', // All Caps for attention but menu item uses logic above
        'manage_options',
        'blyxxa-store',
        'blyxxa_store_page'
    );
     // Update the main menu item if it's the Store page (optional, but sticking to submenu for now)
    global $submenu;
    if (isset($submenu['blyxxa-hq'])) {
        foreach ($submenu['blyxxa-hq'] as $key => $val) {
            if ($val[2] === 'blyxxa-store') {
                $submenu['blyxxa-hq'][$key][0] = $store_title;
            }
        }
    }
});

// Helper: Get Update Count
function blyxxa_get_store_updates() {
    $updates = get_transient('blyxxa_store_updates_count');
    if ($updates === false) {
        // Re-calculate if not cached
        // This is a lightweight check, we don't want to fetch remote every time
        // We rely on the cache set by blyxxa-store.php
        $updates = 0;
    }
    return intval($updates);
}

// Settings (API Keys) & About Us
add_action('admin_menu', function() {
    add_submenu_page(
        'blyxxa-hq',
        __('Settings', 'blyxxa-core'),
        __('Settings', 'blyxxa-core'),
        'manage_options',
        'blyxxa-settings',
        'blyxxa_api_form_localized'
    );

    // About Us
    add_submenu_page(
        'blyxxa-hq',
        __('About Us', 'blyxxa-core'),
        __('About Us', 'blyxxa-core'),
        'manage_options',
        'blyxxa-about',
        'blyxxa_about_page'
    );
});


function blyxxa_about_page() {
    ?>
    <div class="wrap" style="font-family:'Inter', sans-serif; max-width:800px; margin:40px auto; background:#fff; padding:40px; border-radius:12px; box-shadow:0 4px 12px rgba(0,0,0,0.05);">
        <div style="text-align:center; margin-bottom:40px;">
            <h1 style="font-size:32px; color:#0f172a; margin-bottom:10px;"><?php esc_html_e('Blyxxa Labs', 'blyxxa-core'); ?></h1>
            <p style="font-size:18px; color:#64748b;"><?php esc_html_e('Empowering the next generation of AI entrepreneurs.', 'blyxxa-core'); ?></p>
        </div>
        
        <div style="margin-bottom:30px;">
            <h2 style="color:#0f172a; border-bottom:2px solid #f1f5f9; padding-bottom:10px;"><?php esc_html_e('Who We Are', 'blyxxa-core'); ?></h2>
            <p style="line-height:1.8; color:#334155; font-size:15px;">
                <?php esc_html_e('Blyxxa Labs is a pioneering software company dedicated to democratizing Artificial Intelligence. We build powerful, modular, and easy-to-use tools that allow anyone to launch their own AI-powered SaaS business in minutes, not months.', 'blyxxa-core'); ?>
            </p>
        </div>
        
        <div style="margin-bottom:30px;">
            <h2 style="color:#0f172a; border-bottom:2px solid #f1f5f9; padding-bottom:10px;"><?php esc_html_e('Our Mission', 'blyxxa-core'); ?></h2>
            <p style="line-height:1.8; color:#334155; font-size:15px;">
                <?php esc_html_e('Our mission is to remove the technical barriers to entry for AI entrepreneurship. We believe that creativity and business acumen should be the only limits to success, not coding skills.', 'blyxxa-core'); ?>
            </p>
        </div>
        
        <div style="display:grid; grid-template-columns:1fr 1fr; gap:20px; margin-top:40px;">
            <div style="background:#f8fafc; padding:20px; border-radius:8px; text-align:center;">
                <h3 style="margin-top:0; color:#06b6d4;"><?php esc_html_e('Contact Us', 'blyxxa-core'); ?></h3>
            <p style="color:#64748b; font-size:14px;"><?php esc_html_e('Have questions or need support?', 'blyxxa-core'); ?></p>
                <a href="mailto:support@blyxxa.com" class="button button-primary"><?php esc_html_e('Email Support', 'blyxxa-core'); ?></a>
            </div>
            <div style="background:#f8fafc; padding:20px; border-radius:8px; text-align:center;">
                <h3 style="margin-top:0; color:#06b6d4;"><?php esc_html_e('Website', 'blyxxa-core'); ?></h3>
            <p style="color:#64748b; font-size:14px;"><?php esc_html_e('Visit our official website.', 'blyxxa-core'); ?></p>
                <a href="https://blyxxa.com" target="_blank" class="button">blyxxa.com</a>
            </div>
        </div>
        
        <div style="text-align:center; margin-top:50px; color:#94a3b8; font-size:12px;">
            &copy; <?php echo esc_html(gmdate('Y')); ?> Blyxxa LLC. All rights reserved.
        </div>
    </div>
    <?php
}

// CLIENT MODE: Include License Manager
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-client-license.php';

// CLIENT MODE: Include Auto-Updater
require_once plugin_dir_path(__FILE__) . 'includes/blyxxa-updater.php';
new Blyxxa_Updater(__FILE__);

// =================================================================
// AJAX HANDLERS
// =================================================================

// TOOL MANAGER: Toggle Status (Draft <-> Published)
add_action('wp_ajax_blyxxa_toggle_tool_status', function() {
    check_ajax_referer('blyx_status_toggle', 'nonce');
    if(!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
    
    $slug = isset($_POST['slug']) ? sanitize_text_field(wp_unslash($_POST['slug'])) : '';
    if(!$slug) wp_send_json_error('Missing slug');
    
    $lib = get_option('blyxxa_tool_library', []);
    
    // Auto-create entry if missing but file exists (Self-Healing)
    if(!isset($lib[$slug])) {
        $file_path = BLYXXA_TOOLS_PATH . $slug . '.php';
        if(file_exists($file_path)) {
            $lib[$slug] = ['status' => 'draft', 'version' => '1.0']; // Default
        } else {
            wp_send_json_error('Tool not found');
        }
    }
    
    if(isset($lib[$slug])) {
        $current = isset($lib[$slug]['status']) ? $lib[$slug]['status'] : 'draft';
        $new_status = ($current === 'published') ? 'draft' : 'published';
        $lib[$slug]['status'] = $new_status;
        update_option('blyxxa_tool_library', $lib);
        
        wp_send_json_success(['status' => $new_status]);
    } else {
         wp_send_json_error('Tool not found');
    }
});