<?php
// BLYXXA CLIENT STORE
// Location: client-dist/blyxxa-core/admin/blyxxa-store.php

function blyxxa_store_page() {
    // 1. CLEAR CACHE ACTION
    if (isset($_GET['refresh_store']) && check_admin_referer('blyx_refresh_store')) {
        delete_transient('blyxxa_store_data_v1');
        echo '<div class="notice notice-success is-dismissible"><p style="color:#0f172a;">✅ ' . esc_html__('Store cache cleared!', 'blyxxa-core') . '</p></div>';
    }

    // 2. FETCH DATA (Cache for 48 Hours)
    $store_data = get_transient('blyxxa_store_data_v1');
    if (!$store_data) {
        $master_url = BLYXXA_MASTER_API_URL . '/store';
        $resp = wp_remote_get($master_url, ['timeout' => 15, 'sslverify' => false]);
        
        if (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200) {
            $body = json_decode(wp_remote_retrieve_body($resp), true);
            if (!empty($body['success']) && !empty($body['tools'])) {
                $store_data = $body['tools'];
                set_transient('blyxxa_store_data_v1', $store_data, 48 * HOUR_IN_SECONDS);
            }
        }
    }

    if (!is_array($store_data)) $store_data = [];

    // 3. ANALYZE LOCAL TOOLS (For Installed/Update checks)
    $local_tools = [];
    $files = glob(BLYXXA_TOOLS_PATH . '*.php');
    if ($files) {
        foreach ($files as $file) {
            $slug = basename($file, '.php');
            $data = get_file_data($file, ['UID' => 'Tool UID', 'Version' => 'Version']);
            
            // We index by UID if available, otherwise by Slug (legacy)
            $uid = !empty($data['UID']) ? $data['UID'] : $slug; // Fallback to slug if no UID
            
            $local_tools[$uid] = [
                'file' => $file,
                'version' => !empty($data['Version']) ? $data['Version'] : '1.0'
            ];
            
            // Also map by ID (slug) just in case UID is missing in store but present in file or vice versa logic
            // But relying on UID match is best practice going forward.
            // For Store display, we match Store Tool UID vs Local Tool UID.
        }
    }
    
    // 4. CALCULATE UPDATES & SAVE TO TRANSIENT
    $updates_count = 0;
    if(!empty($store_data)) {
        foreach($store_data as $tool) {
            $tuid = $tool['tool_uid'];
            $tid = $tool['id'];
            $ver_remote = $tool['version'];
            
            $local_ver = '0';
            if (isset($local_tools[$tuid])) {
                $local_ver = $local_tools[$tuid]['version'];
                if (version_compare($ver_remote, $local_ver, '>')) $updates_count++;
            } elseif (isset($local_tools[$tid])) {
                $local_ver = $local_tools[$tid]['version'];
                if (version_compare($ver_remote, $local_ver, '>')) $updates_count++;
            }
        }
    }
    set_transient('blyxxa_store_updates_count', $updates_count, 12 * HOUR_IN_SECONDS);

    // 5. DATA FILTERING FOR TABS
    $current_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'store';
    $display_data = $store_data;

    // Filter for Downloaded Tab
    if ($current_tab === 'downloaded') {
        $display_data = array_filter($store_data, function($t) use ($local_tools) {
            return isset($local_tools[$t['tool_uid']]) || isset($local_tools[$t['id']]);
        });
        
        // Sort: Updates First
        usort($display_data, function($a, $b) use ($local_tools) {
            // Check Update Status A
            $a_needs_update = false;
            $a_local = isset($local_tools[$a['tool_uid']]) ? $local_tools[$a['tool_uid']] : (isset($local_tools[$a['id']]) ? $local_tools[$a['id']] : null);
            if($a_local && version_compare($a['version'], $a_local['version'], '>')) $a_needs_update = true;

            // Check Update Status B
            $b_needs_update = false;
            $b_local = isset($local_tools[$b['tool_uid']]) ? $local_tools[$b['tool_uid']] : (isset($local_tools[$b['id']]) ? $local_tools[$b['id']] : null);
            if($b_local && version_compare($b['version'], $b_local['version'], '>')) $b_needs_update = true;

            if ($a_needs_update && !$b_needs_update) return -1;
            if (!$a_needs_update && $b_needs_update) return 1;
            return 0; 
        });
    }

    // 6. CATEGORIES FOR FILTER (Based on Full Data)
    $categories = [];
    if(!empty($store_data)) {
        foreach($store_data as $tool) {
            $cat = !empty($tool['cat']) ? $tool['cat'] : 'General';
            if(!in_array($cat, $categories)) {
                $categories[] = $cat;
            }
        }
    }
    sort($categories);
    ?>

    <div class="wrap" style="font-family:'Inter', sans-serif; max-width:1200px; margin: 20px auto;">
        
        <!-- HEADER -->
        <div style="margin-bottom: 25px; background:linear-gradient(135deg, #0f172a 0%, #1e293b 100%); padding:30px; border-radius:16px; color:#fff; display:flex; justify-content:space-between; align-items:center; box-shadow:0 10px 25px -5px rgba(15, 23, 42, 0.2);">
            <div>
                <h1 style="color:#fff; margin:0 0 10px 0; font-size:28px; display:flex; align-items:center; gap:12px;">
                    <span class="dashicons dashicons-store" style="font-size:32px; width:32px; height:32px; color:#22d3ee;"></span>
                    <?php esc_html_e('Blyxxa Tool Store', 'blyxxa-core'); ?>
                </h1>
                <p style="margin:0; opacity:0.8; font-size:15px; max-width:600px;">
                    <?php esc_html_e('Browse and download ready-made AI tools. All downloaded tools are fully editable and yours to customize!', 'blyxxa-core'); ?>
                </p>
                
                <div style="margin-top:15px; display:inline-flex; align-items:center; background:rgba(255,255,255,0.1); padding:6px 15px; border-radius:20px; font-size:13px; border:1px solid rgba(255,255,255,0.2);">
                    <span class="dashicons dashicons-yes" style="color:#4ade80; margin-right:6px;"></span> 
                    <?php esc_html_e('Unlimited Downloads', 'blyxxa-core'); ?> 
                    <span style="margin:0 10px; opacity:0.3;">|</span>
                    <span class="dashicons dashicons-admin-customizer" style="color:#fcd34d; margin-right:6px;"></span>
                    <?php esc_html_e('Fully Editable Code', 'blyxxa-core'); ?>
                </div>
            </div>
            
            <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=blyxxa-store&refresh_store=1'), 'blyx_refresh_store')); ?>" class="button" style="background:rgba(255,255,255,0.15); color:#fff; border:none; padding:10px 20px; height:auto; display:flex; align-items:center; gap:8px; transition:0.2s;">
                <span class="dashicons dashicons-update"></span> <?php esc_html_e('Check for New Tools', 'blyxxa-core'); ?>
            </a>
        </div>

        <!-- TABS -->
        <h2 class="nav-tab-wrapper" style="margin-bottom:20px; border-bottom:1px solid #cbd5e1;">
            <a href="<?php echo esc_url(admin_url('admin.php?page=blyxxa-store&tab=store')); ?>" class="nav-tab <?php echo ($current_tab !== 'downloaded') ? 'nav-tab-active' : ''; ?>" style="font-size:15px; padding:10px 20px;">
                <span class="dashicons dashicons-grid-view" style="margin-right:5px; vertical-align:middle;"></span> <?php esc_html_e('Store', 'blyxxa-core'); ?>
            </a>
            <a href="<?php echo esc_url(admin_url('admin.php?page=blyxxa-store&tab=downloaded')); ?>" class="nav-tab <?php echo ($current_tab === 'downloaded') ? 'nav-tab-active' : ''; ?>" style="font-size:15px; padding:10px 20px;">
                <span class="dashicons dashicons-download" style="margin-right:5px; vertical-align:middle;"></span> <?php esc_html_e('Downloaded', 'blyxxa-core'); ?>
                <?php if($updates_count > 0): ?>
                    <span style="background:#ef4444; color:#fff; border-radius:50%; padding:2px 6px; font-size:10px; vertical-align:top; margin-left:5px;"><?php echo intval($updates_count); ?></span>
                <?php endif; ?>
            </a>
        </h2>
        
        <!-- FILTER BAR -->
        <div style="background:#fff; padding:15px 20px; border-radius:12px; border:1px solid #e2e8f0; margin-bottom:20px; display:flex; gap:15px; align-items:center;">
            <div style="flex:1; position:relative;">
                <span class="dashicons dashicons-search" style="position:absolute; left:12px; top:50%; transform:translateY(-50%); color:#94a3b8;"></span>
                <input type="text" id="store_search" placeholder="<?php esc_attr_e('Search tools...', 'blyxxa-core'); ?>" onkeyup="filterStore()" style="width:100%; padding:10px 10px 10px 35px; border:1px solid #cbd5e1; border-radius:8px; font-size:14px;">
            </div>
            
            <select id="store_cat_filter" onchange="filterStore()" style="padding:10px 15px; border:1px solid #cbd5e1; border-radius:8px; font-size:14px; min-width:150px; background:#fff;">
                <option value=""><?php esc_html_e('All Categories', 'blyxxa-core'); ?></option>
                <?php foreach($categories as $cat) : ?>
                    <option value="<?php echo esc_attr($cat); ?>"><?php echo esc_html($cat); ?></option>
                <?php endforeach; ?>
            </select>
            
            <?php if($current_tab === 'downloaded'): ?>
                <div style="display:flex; align-items:center; gap:5px; margin-left:10px;">
                    <label style="font-size:13px; color:#64748b; font-weight:600;">
                        <input type="checkbox" id="show-updates-only" onchange="filterStore()"> <?php esc_html_e('Updates Available', 'blyxxa-core'); ?>
                    </label>
                </div>
            <?php endif; ?>
        </div>

        <!-- GRID -->
        <div id="store_grid" style="display:grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap:20px;">
            <?php if(empty($display_data)): ?>
                <div style="grid-column:1/-1; text-align:center; padding:50px; background:#f8fafc; border-radius:12px; color:#64748b;">
                    <span class="dashicons dashicons-download" style="font-size:48px; width:48px; height:48px; color:#cbd5e1; margin-bottom:15px;"></span>
                    <h3 style="margin:0; font-size:18px; color:#334155;"><?php esc_html_e('No downloaded tools found.', 'blyxxa-core'); ?></h3>
                </div>
            <?php else: ?>
            <?php foreach($display_data as $tool) : 
                $tid = $tool['id'];
                $tuid = $tool['tool_uid'];
                $ver_remote = $tool['version'];
                
                // State Logic
                $state = 'new'; // new, installed, update
                $local_ver = '0';
                
                if (isset($local_tools[$tuid])) {
                    // Match by UID (Preferred)
                    $local_ver = $local_tools[$tuid]['version'];
                    // Compare versions
                    if (version_compare($ver_remote, $local_ver, '>')) {
                        $state = 'update';
                    } else {
                        $state = 'installed';
                    }
                } elseif (isset($local_tools[$tid])) {
                    // Match by Slug (Fallback)
                    $local_ver = $local_tools[$tid]['version'];
                    if (version_compare($ver_remote, $local_ver, '>')) {
                        $state = 'update';
                    } else {
                        $state = 'installed';
                    }
                }
                
                // Category Badge Color
                $cat_char = strtoupper(substr($tool['cat'] ?: 'G', 0, 1));
                $colors = ['A'=>'#ef4444', 'B'=>'#f97316', 'C'=>'#f59e0b', 'D'=>'#84cc16', 'E'=>'#10b981', 'F'=>'#06b6d4', 'G'=>'#3b82f6', 'M'=>'#a855f7', 'S'=>'#ec4899'];
                $badge_color = isset($colors[$cat_char]) ? $colors[$cat_char] : '#64748b';
            ?>
            <div class="blyx-store-card" data-title="<?php echo esc_attr(strtolower($tool['title'])); ?>" data-cat="<?php echo esc_attr($tool['cat']); ?>" data-update="<?php echo ($state === 'update') ? '1' : '0'; ?>" style="background:#fff; border:1px solid #e2e8f0; border-radius:12px; overflow:hidden; transition:transform 0.2s; display:flex; flex-direction:column;">
                
                <div style="padding:20px; border-bottom:1px solid #f1f5f9; flex:1;">
                    <div style="display:flex; justify-content:space-between; align-items:start; margin-bottom:12px;">
                        <span style="background:<?php echo esc_attr($badge_color); ?>; color:#fff; padding:3px 8px; border-radius:6px; font-size:11px; font-weight:600; text-transform:uppercase;">
                            <?php echo esc_html($tool['cat'] ?: 'General'); ?>
                        </span>
                        <!-- Version hidden from user generally but useful for debug -->
                        <!-- <span style="font-size:10px; color:#cbd5e1;">v<?php echo esc_html($ver_remote); ?></span> -->
                    </div>
                    
                    <h3 style="margin:0 0 10px 0; color:#1e293b; font-size:16px; font-weight:700;"><?php echo esc_html($tool['title']); ?></h3>
                    <p style="margin:0; color:#64748b; font-size:13px; line-height:1.5;">
                        <?php echo esc_html($tool['desc']); ?>
                    </p>
                    
                    <div style="margin-top:15px; display:flex; gap:10px; font-size:12px; color:#94a3b8;">
                        <span title="Inputs"><span class="dashicons dashicons-forms" style="font-size:14px; vertical-align:middle;"></span> <?php echo esc_html($tool['inputs_count']); ?> Inputs</span>
                    </div>
                </div>
                
                <div style="padding:15px; background:#f8fafc; border-top:1px solid #f1f5f9;">
                    <?php if($state === 'installed') : ?>
                        <button disabled class="button" style="width:100%; display:flex; justify-content:center; align-items:center; gap:5px; background:#e2e8f0; color:#64748b; border:none;">
                            <span class="dashicons dashicons-yes"></span> <?php esc_html_e('Installed', 'blyxxa-core'); ?>
                        </button>
                    <?php elseif($state === 'update') : ?>
                        <button onclick="downloadTool('<?php echo esc_js($tid); ?>', this)" class="button button-primary" style="width:100%; background:#f59e0b; border-color:#f59e0b; display:flex; justify-content:center; align-items:center; gap:5px;">
                            <span class="dashicons dashicons-update"></span> <?php esc_html_e('Update Available', 'blyxxa-core'); ?>
                        </button>
                    <?php else : ?>
                        <button onclick="downloadTool('<?php echo esc_js($tid); ?>', this)" class="button button-primary" style="width:100%; display:flex; justify-content:center; align-items:center; gap:5px;">
                            <span class="dashicons dashicons-download"></span> <?php esc_html_e('Download', 'blyxxa-core'); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
            <?php endif; ?>
            
        </div>
        
        <!-- TOOL REQUEST BANNER -->
        <div style="background:#f0f9ff; border:1px solid #b9e6fe; border-radius:12px; padding:25px; margin-top:30px; display:flex; justify-content:space-between; align-items:center; gap:20px; flex-wrap:wrap;">
            <div style="display:flex; align-items:center; gap:15px;">
                <div style="width:50px; height:50px; background:#e0f2fe; border-radius:50%; display:flex; align-items:center; justify-content:center; color:#0284c7;">
                    <span class="dashicons dashicons-lightbulb" style="font-size:28px; width:28px; height:28px;"></span>
                </div>
                <div>
                    <h3 style="margin:0 0 5px 0; color:#0c4a6e; font-size:16px;"><?php esc_html_e('Did not find the tool you are looking for?', 'blyxxa-core'); ?></h3>
                    <p style="margin:0; color:#334155; font-size:14px; max-width:600px;">
                        <?php esc_html_e('Request it from the support section, if suitable we will add it to the store for you within 5 business days.', 'blyxxa-core'); ?>
                    </p>
                </div>
            </div>
            <a href="<?php echo esc_url(admin_url('admin.php?page=blyxxa-hq&tab=support')); ?>" class="button" style="background:#0284c7; color:#fff; border:none; padding:12px 25px; height:auto; font-size:14px; font-weight:600; display:flex; align-items:center; gap:8px; transition:0.2s; white-space:nowrap;">
                <?php esc_html_e('Request a Tool', 'blyxxa-core'); ?> <span class="dashicons dashicons-arrow-right-alt2" style="margin-top:2px;"></span>
            </a>
        </div>
        
    </div>

    <!-- JS Logic -->
    <script>
    function filterStore() {
        let txt = document.getElementById('store_search').value.toLowerCase();
        let cat = document.getElementById('store_cat_filter').value;
        let cards = document.querySelectorAll('.blyx-store-card');
        
        cards.forEach(c => {
            let t = c.getAttribute('data-title');
            let cCat = c.getAttribute('data-cat');
            let isUpdate = c.getAttribute('data-update') === '1';
            
            let matchTxt = t.includes(txt);
            let matchCat = (cat === '' || cCat === cat);
            
            // Updates Only Checkbox
            let updatesOnly = false;
            let chk = document.getElementById('show-updates-only');
            if(chk && chk.checked) {
                updatesOnly = true;
            }
            
            let matchUpdate = (!updatesOnly || isUpdate);
            
            c.style.display = (matchTxt && matchCat && matchUpdate) ? 'flex' : 'none';
        });
    }

    // Custom Toast Function
    function blyxxaToast(msg, type='success') {
        // Remove existing
        let exist = document.getElementById('blyx-toast');
        if(exist) exist.remove();
        
        // Create new
        let el = document.createElement('div');
        el.id = 'blyx-toast';
        el.style.cssText = 'position:fixed; bottom:20px; right:20px; background:#1e293b; color:#fff; padding:12px 20px; border-radius:8px; z-index:99999; display:flex; align-items:center; gap:10px; box-shadow:0 10px 25px rgba(0,0,0,0.2); animation: slideIn 0.3s ease-out forwards;';
        
        let icon = (type === 'error') ? 'dashicons-warning' : 'dashicons-yes';
        let color = (type === 'error') ? '#ef4444' : '#4ade80';
        
        el.innerHTML = '<span class="dashicons ' + icon + '" style="color:' + color + '"></span> ' + msg;
        
        document.body.appendChild(el);
        
        setTimeout(() => {
            el.style.opacity = '0';
            el.style.transform = 'translateY(10px)';
            el.style.transition = '0.3s';
            setTimeout(() => el.remove(), 300);
        }, 3000);
    }

    function downloadTool(toolId, btn) {
        let originalHtml = btn.innerHTML;
        btn.innerHTML = '<span class="dashicons dashicons-update" style="animation:spin 1s infinite linear;"></span> <?php echo esc_js(__('Downloading...', 'blyxxa-core')); ?>';
        btn.disabled = true;
        
        let data = new FormData();
        data.append('action', 'blyxxa_client_download_tool');
        data.append('tool_id', toolId);
        data.append('nonce', '<?php echo esc_js(wp_create_nonce('blyx_client_dl')); ?>');
        
        fetch(ajaxurl, {method:'POST', body:data}).then(r=>r.json()).then(res => {
            if(res.success) {
                // Success
                btn.className = 'button';
                btn.style.background = '#10b981';
                btn.style.borderColor = '#10b981';
                btn.style.color = '#fff';
                btn.innerHTML = '<span class="dashicons dashicons-yes"></span> <?php echo esc_js(__('Installed!', 'blyxxa-core')); ?>';
                
                // Show success toast
                if(typeof Swal !== 'undefined') {
                    Swal.fire({
                        icon: 'success', 
                        title: '<?php echo esc_js(__('Tool Installed', 'blyxxa-core')); ?>',
                        text: '<?php echo esc_js(__('You can now customize it in Tool Manager.', 'blyxxa-core')); ?>',
                        timer: 2000,
                        showConfirmButton: false,
                        toast: true,
                        position: 'bottom-end'
                    });
                } else {
                    blyxxaToast('<?php echo esc_js(__('Tool installed successfully!', 'blyxxa-core')); ?>', 'success');
                }
            } else {
                // Error
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                if(typeof Swal !== 'undefined') Swal.fire('Error', res.data, 'error');
                else blyxxaToast(res.data, 'error');
            }
        }).catch(e => {
            btn.innerHTML = originalHtml;
            btn.disabled = false;
            blyxxaToast('Connection Error', 'error');
        });
    }
    </script>
    <style>
        .blyx-store-card:hover { transform: translateY(-3px); box-shadow: 0 10px 20px -5px rgba(0,0,0,0.1) !important; }
        @keyframes spin { 100% { transform:rotate(360deg); } }
        @keyframes slideIn { from { transform:translateY(20px); opacity:0; } to { transform:translateY(0); opacity:1; } }
    </style>
    <?php
}
