<?php
/**
 * Master Tool Template
 * 
 * This file handles the rendering and logic for all tools generated by the Blyxxa Factory.
 * It expects a $tool_config array to be defined before inclusion.
 * 
 * @var array $tool_config {
 *     @type string $id       Unique tool ID (snake_case)
 *     @type string $title    Tool Name
 *     @type string $desc     Tool Description
 *     @type string $prompt   System Prompt
 *     @type array  $inputs   Array of input definitions
 * }
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// Extract config for easier access
// Handle both prefixed (new) and non-prefixed (legacy) variable names
$blyxxa_final_config = isset($blyxxa_tool_config) ? $blyxxa_tool_config : (isset($tool_config) ? $tool_config : []);

if (empty($blyxxa_final_config)) {
    return 'Tool configuration missing.';
}

$blyxxa_t_id = $blyxxa_final_config['id'] ?? '';
$blyxxa_t_title = $blyxxa_final_config['title'] ?? '';
$blyxxa_t_desc = $blyxxa_final_config['desc'] ?? '';
$blyxxa_t_inputs = $blyxxa_final_config['inputs'] ?? [];

// LICENSE & DOWNGRADE CHECK
$blyxxa_is_allowed = true;
if (class_exists('Blyxxa_Client_License')) {
    $blyxxa_lic_status = Blyxxa_Client_License::get_license_status();
    if ($blyxxa_lic_status !== 'active') {
        // License is NOT active (Free/Expired) -> Allow only 1 tool (Oldest)
        $blyxxa_all_tools = glob(BLYXXA_TOOLS_PATH . '*.php');
        if($blyxxa_all_tools) {
            // Sort by creation time ASC (Oldest first)
            usort($blyxxa_all_tools, function($a, $b) {
                return filectime($a) - filectime($b);
            });
            
            $blyxxa_allowed_file = $blyxxa_all_tools[0];
            $blyxxa_current_file = BLYXXA_TOOLS_PATH . $blyxxa_final_config['id'] . '.php'; // Reconstruct path
            
            // Loose comparison of filenames or strict path check
            if (realpath($blyxxa_current_file) !== realpath($blyxxa_allowed_file)) {
                $blyxxa_is_allowed = false;
            }
        }
    }
}

if (!$blyxxa_is_allowed) {
    return '<div style="background:#fef2f2; border:1px solid #fecaca; padding:20px; border-radius:8px; color:#991b1b; text-align:center; font-family:sans-serif;">' . 
           '<strong>' . esc_html__('Tool Unavailable', 'blyxxa-core') . '</strong><br>' . 
           esc_html__('This tool is currently inactive due to the site\'s subscription plan limits.', 'blyxxa-core') . 
           '</div>';
}
add_shortcode('tool_' . $blyxxa_t_id, function() use ($blyxxa_t_id, $blyxxa_t_title, $blyxxa_t_desc, $blyxxa_t_inputs) {
    ob_start();
    
    // Fix: Map variables to short names used in template
    $t_id = $blyxxa_t_id;
    $t_title = $blyxxa_t_title;
    $t_desc = $blyxxa_t_desc;
    $t_inputs = $blyxxa_t_inputs;
    
    // Brand Colors (Could be dynamic in future)
    $brand_cyan = '#06b6d4'; 
    $brand_dark = '#0f172a';
    ?>
    <?php
    // Enqueue External Libraries for Rendering
    wp_enqueue_script('marked-js', 'https://cdn.jsdelivr.net/npm/marked/marked.min.js', [], '4.0', true);
    wp_enqueue_script('mathjax-js', 'https://cdn.jsdelivr.net/npm/mathjax@3/es5/tex-mml-chtml.js', [], '3.0', true);
    wp_enqueue_script('sweetalert2', 'https://cdn.jsdelivr.net/npm/sweetalert2@11', [], '11.0', true); // Added SweetAlert2
    
    // Add MathJax Configuration as inline script (before MathJax loads)
    wp_add_inline_script('mathjax-js', '
        window.MathJax = {
            tex: { inlineMath: [["$", "$"], ["\\\\(", "\\\\)"]] },
            svg: { fontCache: "global" }
        };
    ', 'before');
    ?>
    <style>
        :root { --brand-cyan: <?php echo esc_attr($brand_cyan); ?>; --brand-dark: <?php echo esc_attr($brand_dark); ?>; }
        .blyx-card { background: #fff; border: 2px solid #cbd5e1; border-top: 5px solid var(--brand-cyan); border-radius: 16px; padding: 45px; max-width: 1100px; margin: 40px auto; box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.15); font-family: 'Inter', system-ui, sans-serif; }
        .blyx-head { text-align: center; margin-bottom: 40px; padding-bottom: 25px; border-bottom: 1px solid #e2e8f0; }
        .blyx-head h2 { margin: 0 0 12px 0; font-size: 36px; color: var(--brand-dark); letter-spacing: -0.8px; font-weight: 800; }
        .blyx-head p { margin: 0; color: #475569; font-size: 18px; line-height: 1.5; }
        .blyx-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 30px; }
        .blyx-col { display: flex; flex-direction: column; }
        .blyx-full { grid-column: 1 / -1; } .blyx-half { grid-column: span 1; }
        @media (max-width: 768px) { .blyx-grid { grid-template-columns: 1fr; } .blyx-half { grid-column: 1 / -1; } }
        .blyx-label { font-weight: 700; font-size: 15px; color: #1e293b; margin-bottom: 10px; display: block; text-transform: uppercase; letter-spacing: 0.6px; }
        .blyx-input, .blyx-select { width: 100%; padding: 18px 20px; border: 2px solid #cbd5e1; border-radius: 12px; background: #fff; box-sizing: border-box; font-size: 16px; color: #0f172a; transition: all 0.2s ease; appearance: none; -webkit-appearance: none; min-height: 60px; line-height: 1.5; }
        .blyx-select { background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%23334155' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M6 8l4 4 4-4'/%3e%3c/svg%3e"); background-position: right 1.5rem center; background-repeat: no-repeat; background-size: 1.5em 1.5em; }
        .blyx-input:focus, .blyx-select:focus { border-color: var(--brand-cyan); outline: none; box-shadow: 0 0 0 4px rgba(6, 182, 212, 0.15); }
        .blyx-chk-wrap { display: flex; align-items: center; gap: 15px; background: #f8fafc; padding: 20px; border: 2px solid #cbd5e1; border-radius: 12px; cursor: pointer; transition: 0.2s; min-height: 60px; }
        .blyx-chk-wrap:hover { border-color: var(--brand-cyan); background: #fff; }
        .blyx-chk-wrap input { width: 24px; height: 24px; accent-color: var(--brand-cyan); cursor: pointer; }
        .blyx-chk-wrap span { font-weight: 700; color: #334155; font-size: 16px; }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
        .blyx-spinner { display: inline-block; width: 24px; height: 24px; border: 3px solid rgba(255,255,255,0.3); border-radius: 50%; border-top-color: #fff; animation: spin 1s ease-in-out infinite; }
        .blyx-btn { width: 100%; padding: 22px; background: var(--brand-cyan); color: #fff; border: none; border-radius: 12px; cursor: pointer; font-weight: 800; font-size: 20px; margin-top: 40px; transition: all 0.2s; box-shadow: 0 15px 30px -5px rgba(6, 182, 212, 0.3); display: flex; justify-content: center; align-items: center; gap: 12px; letter-spacing: 0.5px; }
        .blyx-btn:hover { background: #0891b2; transform: translateY(-3px); box-shadow: 0 20px 40px -10px rgba(6, 182, 212, 0.5); }
        .blyx-btn:disabled { background: #94a3b8; cursor: not-allowed; transform: none; opacity: 0.8; }
        
        /* RESULT AREA STYLES - FLATTENED */
        /* RESULT AREA STYLES - FLATTENED */
        .blyx-res { margin-top: 20px; padding-top: 15px; border-top: 1px solid #e2e8f0; display: none; animation: fadeIn 0.6s cubic-bezier(0.16, 1, 0.3, 1); }
        @keyframes fadeIn { from { opacity: 0; transform: translateY(10px); } to { opacity: 1; transform: translateY(0); } }
        
        /* MARKDOWN CONTENT STYLES */
        .blyx-content { font-size: 17px; line-height: 1.8; color: #334155; overflow-x: auto; }
        .blyx-content h1, .blyx-content h2, .blyx-content h3 { color: var(--brand-dark); margin-top: 1.5em; margin-bottom: 0.8em; font-weight: 800; line-height: 1.3; }
        .blyx-content h1 { font-size: 28px; border-bottom: 2px solid #e2e8f0; padding-bottom: 10px; }
        .blyx-content h2 { font-size: 24px; }
        .blyx-content h3 { font-size: 20px; }
        .blyx-content p { margin-bottom: 1.5em; }
        .blyx-content ul, .blyx-content ol { margin-bottom: 1.5em; padding-left: 25px; }
        .blyx-content li { margin-bottom: 0.5em; }
        .blyx-content strong { color: var(--brand-dark); font-weight: 700; }
        .blyx-content blockquote { border-left: 4px solid var(--brand-cyan); background: #ecfeff; padding: 15px 20px; margin: 20px 0; border-radius: 0 8px 8px 0; font-style: italic; color: #155e75; }
        .blyx-content code { background: #f1f5f9; padding: 2px 6px; border-radius: 4px; font-family: 'Consolas', monospace; font-size: 0.9em; color: #0f172a; }
        .blyx-content pre { background: #1e293b; color: #f8fafc; padding: 20px; border-radius: 8px; overflow-x: auto; margin: 20px 0; }
        .blyx-content pre code { background: none; color: inherit; padding: 0; }
        .blyx-content a { color: var(--brand-cyan); text-decoration: underline; font-weight: 600; }
        
        /* TABLE STYLES */
        .blyx-content table { width: 100%; border-collapse: collapse; margin: 25px 0; font-size: 15px; border-radius: 8px; overflow: hidden; box-shadow: 0 0 0 1px #e2e8f0; }
        .blyx-content th, .blyx-content td { padding: 15px; border: 1px solid #e2e8f0; text-align: left; }
        .blyx-content th { background-color: #f8fafc; font-weight: 700; color: var(--brand-dark); text-transform: uppercase; font-size: 13px; letter-spacing: 0.5px; }
        .blyx-content tr:nth-child(even) { background-color: #fcfcfc; }
        .blyx-content tr:hover { background-color: #f1f5f9; }
        
        .blyx-actions { display: flex; gap: 15px; margin-top: 35px; flex-wrap: wrap; }
        .blyx-act-btn { padding: 14px 28px; border: 2px solid #cbd5e1; background: #fff; border-radius: 10px; cursor: pointer; font-size: 15px; font-weight: 700; color: #475569; display: flex; align-items: center; gap: 10px; transition: all 0.2s; }
        .blyx-act-btn:hover { background: #fff; border-color: var(--brand-cyan); color: var(--brand-dark); transform: translateY(-2px); }
        .blyx-primary-act { background: var(--brand-dark); color: #fff; border-color: var(--brand-dark); }
        .blyx-primary-act:hover { background: var(--brand-cyan); color: #fff; border-color: var(--brand-cyan); }
        .blyx-rev { margin-top: 40px; border-top: 2px dashed #cbd5e1; padding-top: 35px; display: none; }
        .blyx-rev-inner { display: flex; gap: 15px; }
        
        @media print { 
            body * { visibility: hidden; } 
            #out_<?php echo esc_attr($t_id); ?>, #out_<?php echo esc_attr($t_id); ?> * { visibility: visible; } 
            #out_<?php echo esc_attr($t_id); ?> { position: absolute; left: 0; top: 0; width: 100%; padding: 0; border: none; box-shadow: none; } 
            .blyx-content table { page-break-inside: auto; }
            .blyx-content tr { page-break-inside: avoid; page-break-after: auto; }
        }

        /* MOBILE RESPONSIVE FIXES */
        @media (max-width: 768px) {
            .blyx-card { padding: 15px !important; margin: 15px 0 !important; border-radius: 12px !important; width: 100% !important; box-sizing: border-box !important; }
            .blyx-content { padding: 0 !important; font-size: 15px !important; overflow-x: hidden !important; } /* Prevent full content scroll, handle table separately */
            .blyx-content table { display: block; width: 100%; overflow-x: auto; white-space: nowrap; -webkit-overflow-scrolling: touch; }
            .blyx-head h2 { font-size: 22px !important; }
            .blyx-head p { font-size: 14px !important; }
            .blyx-grid { grid-template-columns: 1fr !important; gap: 15px !important; }
            .blyx-col { width: 100% !important; }
            .blyx-btn { font-size: 16px !important; padding: 15px !important; margin-top: 20px !important; }
            .blyx-res { padding: 15px 0 0 0 !important; margin-top: 20px !important; }
            .blyx-actions { flex-direction: column; }
            .blyx-act-btn { width: 100%; justify-content: center; }
            .blyx-rev-inner { flex-direction: column; }
            .blyx-rev-inner button { width: 100% !important; margin-top: 10px; }
        }
    </style>

    <div class="blyx-card">
        <div class="blyx-head">
            <h2><?php echo esc_html($t_title); ?></h2>
            <p><?php echo esc_html($t_desc); ?></p>
        </div>

        <div class="blyx-grid" style="margin-top:30px;">
            <div class="blyx-full">
                <label class="blyx-label"><?php esc_html_e('AI Model', 'blyxxa-core'); ?></label>
                <select id="mod_<?php echo esc_attr($t_id); ?>" class="blyx-select">
                    <option value="" disabled selected><?php esc_html_e('Initializing AI Models...', 'blyxxa-core'); ?></option>
                </select>
            </div>
        </div>

        <form id="frm_<?php echo esc_attr($t_id); ?>" class="blyx-grid" style="margin-top:30px;">
            <?php wp_nonce_field('blyxxa_tool_nonce', 'nonce'); ?>
            
            <?php foreach($t_inputs as $input): 
                $name = $input['name'];
                // Ensure numeric names have prefix
                if(is_numeric(substr($name, 0, 1))) { $name = 'val_' . $name; }
                
                $label = $input['label'];
                $type = $input['type'];
                $ph = isset($input['ph']) ? esc_attr($input['ph']) : '';
                $col_class = ($type == 'textarea') ? 'blyx-full' : 'blyx-half';
            ?>
                <div class="blyx-col <?php echo esc_attr($col_class); ?>">
                    <?php if($type == 'checkbox'): ?>
                        <label class="blyx-chk-wrap">
                            <input type="checkbox" name="<?php echo esc_attr($name); ?>" value="1"> 
                            <span><?php echo esc_html($label); ?></span>
                        </label>
                    <?php elseif($type == 'file'): ?>
                        <label class="blyx-label"><?php echo esc_html($label); ?></label>
                        <input type="file" name="<?php echo esc_attr($name); ?>" class="blyx-input" accept="image/png, image/jpeg, image/webp, application/pdf">
                        <small style="color:#64748b; font-size:12px;"><?php echo $ph ? esc_html($ph) : esc_html__('Upload an image or document to analyze.', 'blyxxa-core'); ?></small>
                    <?php else: ?>
                        <label class="blyx-label"><?php echo esc_html($label); ?></label>
                        <?php if($type == 'text'): ?>
                            <input type="text" name="<?php echo esc_attr($name); ?>" class="blyx-input" placeholder="<?php echo esc_attr($ph); ?>">
                        <?php elseif($type == 'textarea'): ?>
                            <textarea name="<?php echo esc_attr($name); ?>" rows="4" class="blyx-input" placeholder="<?php echo esc_attr($ph); ?>"></textarea>
                        <?php elseif($type == 'number'): ?>
                            <input type="number" name="<?php echo esc_attr($name); ?>" class="blyx-input" placeholder="<?php echo esc_attr($ph); ?>">
                        <?php elseif($type == 'select'): ?>
                            <select name="<?php echo esc_attr($name); ?>" class="blyx-select">
                                <?php 
                                $opts = preg_split('/[\r\n,]+/', $input['opts'], -1, PREG_SPLIT_NO_EMPTY);
                                foreach($opts as $opt): 
                                    $opt = trim($opt);
                                    if(!empty($opt)): ?>
                                        <option value="<?php echo esc_attr($opt); ?>"><?php echo esc_html($opt); ?></option>
                                    <?php endif; 
                                endforeach; ?>
                            </select>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </form>

        <button id="btn_<?php echo esc_attr($t_id); ?>" onclick="run_<?php echo esc_js($t_id); ?>()" class="blyx-btn">
            <span id="btn_icon_<?php echo esc_attr($t_id); ?>" class="dashicons dashicons-rocket" style="font-size:28px;width:28px;height:28px;"></span>
            <span id="btn_text_<?php echo esc_attr($t_id); ?>"><?php esc_html_e('GENERATE STRATEGY', 'blyxxa-core'); ?></span>
        </button>

        <div id="res_<?php echo esc_attr($t_id); ?>" class="blyx-res">
            <div id="out_<?php echo esc_attr($t_id); ?>" class="blyx-content"></div>
            
            <div class="blyx-actions">
                <button onclick="save_<?php echo esc_attr($t_id); ?>()" class="blyx-act-btn blyx-primary-act"><?php esc_html_e('💾 Save to Dashboard', 'blyxxa-core'); ?></button>
                <button onclick="copy_<?php echo esc_attr($t_id); ?>()" class="blyx-act-btn"><?php esc_html_e('📋 Copy', 'blyxxa-core'); ?></button>
                <button onclick="downloadDocx_<?php echo esc_attr($t_id); ?>()" class="blyx-act-btn"><?php esc_html_e('📄 DOCX', 'blyxxa-core'); ?></button>
                <button onclick="downloadXls_<?php echo esc_attr($t_id); ?>()" class="blyx-act-btn"><?php esc_html_e('📊 Excel', 'blyxxa-core'); ?></button>
                <button onclick="print_<?php echo esc_attr($t_id); ?>()" class="blyx-act-btn"><?php esc_html_e('🖨️ Print', 'blyxxa-core'); ?></button>
            </div>

            <div id="rev_box_<?php echo esc_attr($t_id); ?>" class="blyx-rev">
                <label class="blyx-label" style="color:var(--brand-cyan);"><?php esc_html_e('Refine Result', 'blyxxa-core'); ?></label>
                <div class="blyx-rev-inner">
                    <input type="text" id="rev_in_<?php echo esc_attr($t_id); ?>" class="blyx-input" placeholder="<?php esc_attr_e('e.g., Make it more persuasive...', 'blyxxa-core'); ?>">
                    <button onclick="run_<?php echo esc_attr($t_id); ?>(true)" class="blyx-act-btn" style="background:var(--brand-cyan); color:#fff; border-color:var(--brand-cyan); flex-shrink:0; font-weight:800;"><?php esc_html_e('Update', 'blyxxa-core'); ?></button>
                </div>
            </div>
        </div>
    </div>

    <?php
    // Branding Footer (Only for Free/Inactive Users)
    if (class_exists('Blyxxa_Client_License') && Blyxxa_Client_License::get_license_status() !== 'active') {
        ?>
        <div class="blyx-branding" style="text-align:center; margin-top:30px; padding-bottom:20px;">
            <a href="https://blyxxa.online/" target="_blank" style="text-decoration:none; color:#94a3b8; font-size:12px; font-weight:600; font-family:'Inter', sans-serif; display:inline-flex; align-items:center; gap:6px; transition:color 0.2s;" onmouseover="this.style.color='#06b6d4'" onmouseout="this.style.color='#94a3b8'">
                <span style="opacity:0.6;">⚡</span> <?php esc_html_e('Powered by Blyxxa SaaS Engine', 'blyxxa-core'); ?>
            </a>
        </div>
        <?php
    }
    ?>

    <script>
    // Initialize Models
    (function() {
        let s = document.getElementById('mod_<?php echo esc_js($t_id); ?>');
        if(window.blyxxaModels && window.blyxxaModels.length > 0) {
            s.innerHTML = '';
            window.blyxxaModels.forEach(m => {
                let opt = new Option(m.name, m.id);
                s.appendChild(opt);
            });
        } else {
            // Fallback fetch if global models not ready
            let d = new FormData(); d.append('action', 'blyxxa_fetch_models');
            fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {method:'POST', body:d})
            .then(r=>r.json()).then(res => {
                if(res.success) {
                    s.innerHTML = '<option value="" disabled selected><?php echo esc_js(__('Select AI Model', 'blyxxa-core')); ?></option>';
                    // Handle Categorized Object
                    if(typeof res.data === 'object' && !Array.isArray(res.data)) {
                        for (const [provider, models] of Object.entries(res.data)) {
                            if(models.length > 0) {
                                let group = document.createElement('optgroup');
                                group.label = provider.charAt(0).toUpperCase() + provider.slice(1);
                                models.forEach(m => {
                                    group.appendChild(new Option(m.name, m.id));
                                });
                                s.appendChild(group);
                            }
                        }
                    } else if (Array.isArray(res.data)) {
                        // Fallback for flat array
                        res.data.forEach(m => { s.appendChild(new Option(m.name, m.id)); });
                    }
                }
            });
        }
    })();

    function run_<?php echo esc_js($t_id); ?>(isRev = false) {
        // Offline Check
        if (!navigator.onLine) {
            Swal.fire({
                icon: 'warning',
                title: '<?php echo esc_js(__('Connection Lost', 'blyxxa-core')); ?>',
                text: '<?php echo esc_js(__('AI tools require an internet connection.', 'blyxxa-core')); ?>',
                confirmButtonColor: 'var(--brand-cyan)'
            });
            return;
        }

        let btn = document.getElementById('btn_<?php echo esc_js($t_id); ?>');
        let icon = document.getElementById('btn_icon_<?php echo esc_js($t_id); ?>');
        let txt = document.getElementById('btn_text_<?php echo esc_js($t_id); ?>');
        let resBox = document.getElementById('res_<?php echo esc_js($t_id); ?>');
        let out = document.getElementById('out_<?php echo esc_js($t_id); ?>');
        let form = document.getElementById('frm_<?php echo esc_js($t_id); ?>');
        
        let model = document.getElementById('mod_<?php echo esc_js($t_id); ?>').value;
        if(!model) { 
            Swal.fire({
                icon: 'info',
                title: '<?php echo esc_js(__('Model Missing', 'blyxxa-core')); ?>',
                text: '<?php echo esc_js(__('Please select an AI Model.', 'blyxxa-core')); ?>',
                confirmButtonColor: 'var(--brand-cyan)'
            });
            return; 
        }

        let data = new FormData(form);
        data.append('action', 'blyxxa_run_tool');
        data.append('tool_id', '<?php echo esc_js($t_id); ?>');
        data.append('model', model);
        
        // Handle Revision
        if(isRev) {
            let revInst = document.getElementById('rev_in_<?php echo esc_js($t_id); ?>').value;
            if(!revInst) return;
            data.append('revision_mode', '1');
            data.append('revision_instruction', revInst);
            data.append('previous_result', out.innerText); // Send raw text for context
            
            // UI Update for Rev
            txt.innerText = '<?php echo esc_js(__('UPDATING...', 'blyxxa-core')); ?>';
        } else {
            txt.innerText = '<?php echo esc_js(__('GENERATING...', 'blyxxa-core')); ?>';
            resBox.style.display = 'none';
        }
        
        btn.disabled = true;
        icon.className = 'blyx-spinner';
        
        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
            method: 'POST',
            body: data
        })
        .then(response => response.json())
        .then(res => {
            if(res.success) {
                // RENDER MARKDOWN
                let raw = res.data;
                let html = marked.parse(raw);
                out.innerHTML = html;
                
                // RENDER MATH
                if(window.MathJax) {
                    MathJax.typesetPromise([out]).catch((err) => console.log(err));
                }

                resBox.style.display = 'block';
                document.getElementById('rev_box_<?php echo esc_js($t_id); ?>').style.display = 'block';
                
                // Scroll to result
                resBox.scrollIntoView({behavior: 'smooth'});
            } else {
                Swal.fire({
                    icon: 'error',
                    title: '<?php echo esc_js(__('Generation Failed', 'blyxxa-core')); ?>',
                    html: res.data,
                    confirmButtonColor: 'var(--brand-cyan)'
                });
            }
        })
        .catch(err => {
            Swal.fire({
                icon: 'error',
                title: '<?php echo esc_js(__('System Error', 'blyxxa-core')); ?>',
                text: err,
                confirmButtonColor: 'var(--brand-cyan)'
            });
        })
        .finally(() => {
            btn.disabled = false;
            icon.className = 'dashicons dashicons-rocket';
            txt.innerText = '<?php echo esc_js(__('GENERATE STRATEGY', 'blyxxa-core')); ?>';
        });
    }

    // Helper Functions
    function copy_<?php echo esc_js($t_id); ?>() {
        let content = document.getElementById('out_<?php echo esc_js($t_id); ?>').innerText;
        navigator.clipboard.writeText(content).then(() => {
            const Toast = Swal.mixin({
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 3000,
                timerProgressBar: true
            });
            Toast.fire({
                icon: 'success',
                title: '<?php echo esc_js(__('Copied to clipboard!', 'blyxxa-core')); ?>'
            });
        });
    }

    function save_<?php echo esc_js($t_id); ?>() {
        let content = document.getElementById('out_<?php echo esc_js($t_id); ?>').innerHTML;
        let d = new FormData();
        d.append('action', 'blyxxa_save_result');
        d.append('tool_id', '<?php echo esc_js($t_id); ?>');
        d.append('content', content);
        d.append('nonce', document.querySelector('#frm_<?php echo esc_js($t_id); ?> input[name="nonce"]').value);
        
        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {method:'POST', body:d})
        .then(r=>r.json()).then(res => {
            if(res.success) {
                 Swal.fire({
                    icon: 'success',
                    title: '<?php echo esc_js(__('Saved!', 'blyxxa-core')); ?>',
                    text: '<?php echo esc_js(__('Result saved to your dashboard.', 'blyxxa-core')); ?>',
                    timer: 2000,
                    showConfirmButton: false
                });
            } else {
                 Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: res.data
                });
            }
        });
    }

    function print_<?php echo esc_js($t_id); ?>() { window.print(); }
    
    // PREMIUM EXPORT: DOCX (HTML-based)
    function downloadDocx_<?php echo esc_js($t_id); ?>() {
        let content = document.getElementById('out_<?php echo esc_js($t_id); ?>').innerHTML;
        // Wrap in HTML structure for Word
        let header = "<html xmlns:o='urn:schemas-microsoft-com:office:office' xmlns:w='urn:schemas-microsoft-com:office:word' xmlns='http://www.w3.org/TR/REC-html40'><head><meta charset='utf-8'><title>Export HTML to Word Document with JavaScript</title><style>body{font-family:Arial, sans-serif;} table{border-collapse:collapse;width:100%;} td,th{border:1px solid #000;padding:8px;}</style></head><body>";
        let footer = "</body></html>";
        let sourceHTML = header + content + footer;
        
        let blob = new Blob(['\ufeff', sourceHTML], { type: 'application/msword' });
        let link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = '<?php echo esc_js($t_id); ?>_result.doc';
        link.click();
    }
    
    // PREMIUM EXPORT: EXCEL (HTML-based)
    function downloadXls_<?php echo esc_js($t_id); ?>() {
        let content = document.getElementById('out_<?php echo esc_js($t_id); ?>').innerHTML;
        let template = '<html xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40"><head><!--[if gte mso 9]><xml><x:ExcelWorkbook><x:ExcelWorksheets><x:ExcelWorksheet><x:Name>Sheet1</x:Name><x:WorksheetOptions><x:DisplayGridlines/></x:WorksheetOptions></x:ExcelWorksheet></x:ExcelWorksheets></x:ExcelWorkbook></xml><![endif]--><meta charset="utf-8"></head><body>' + content + '</body></html>';
        
        let blob = new Blob(['\ufeff', template], {type: 'application/vnd.ms-excel'});
        let link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = '<?php echo esc_js($t_id); ?>_result.xls';
        link.click();
    }
    </script>
    <?php
    return ob_get_clean();
});
