<?php
if (!defined('ABSPATH')) exit;

class Blyxxa_Updater {
    private $slug;
    private $plugin_file;
    private $plugin_basename;
    private $api_url;

    public function __construct($plugin_file) {
        $this->plugin_file = $plugin_file;
        $this->plugin_basename = plugin_basename($plugin_file);
        $this->slug = dirname($this->plugin_basename);
        $this->api_url = defined('BLYXXA_MASTER_API_URL') ? BLYXXA_MASTER_API_URL : '';
        
        // Hook into the update transient
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_update']);
        
        // Hook into the plugin details popup
        add_filter('plugins_api', [$this, 'check_info'], 20, 3);
        
        // Clear cache on plugin update
        add_action('upgrader_process_complete', [$this, 'clear_cache'], 10, 2);
    }

    public function check_update($transient) {
        if (!is_object($transient)) {
            $transient = new stdClass();
        }
        
        // Get current version
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $plugin_data = get_plugin_data($this->plugin_file);
        $current_version = $plugin_data['Version'];

        // Check Master Node
        $remote_data = $this->get_remote_version();

        if ($remote_data && isset($remote_data['new_version'])) {
            if (version_compare($current_version, $remote_data['new_version'], '<')) {
                $obj = new stdClass();
                $obj->slug = $this->slug;
                $obj->plugin = $this->plugin_basename; // IMPORTANT: WordPress needs this
                $obj->new_version = $remote_data['new_version'];
                $obj->url = $remote_data['url'] ?? '';
                $obj->package = $remote_data['package'] ?? '';
                $obj->tested = '6.7';
                $obj->requires_php = '7.4';
                $obj->icons = [
                    '1x' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif',
                    '2x' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif',
                    'default' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif'
                ];
                
                if (!isset($transient->response)) {
                    $transient->response = [];
                }
                
                $transient->response[$this->plugin_basename] = $obj;
            } else {
                // No update available - add to no_update list
                if (!isset($transient->no_update)) {
                    $transient->no_update = [];
                }
                
                $obj = new stdClass();
                $obj->slug = $this->slug;
                $obj->plugin = $this->plugin_basename;
                $obj->new_version = $current_version;
                $obj->url = '';
                $obj->package = '';
                
                $transient->no_update[$this->plugin_basename] = $obj;
            }
        }

        return $transient;
    }

    public function check_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }

        if (!isset($args->slug) || $args->slug !== $this->slug) {
            return $result;
        }

        $remote_data = $this->get_remote_version();

        if ($remote_data) {
            $obj = new stdClass();
            $obj->slug = $this->slug;
            $obj->name = __('Blyxxa SaaS Engine', 'blyxxa-core');
            $obj->version = $remote_data['new_version'];
            $obj->new_version = $remote_data['new_version'];
            $obj->requires = '6.0';
            $obj->tested = '6.7';
            $obj->requires_php = '7.4';
            $obj->author = '<a href="https://blyxxa.com">Blyxxa LLC</a>';
            $obj->author_profile = 'https://blyxxa.com';
            $obj->homepage = 'https://blyxxa.com';
            $obj->download_link = $remote_data['package'] ?? '';
            $obj->trunk = $remote_data['package'] ?? '';
            $obj->last_updated = gmdate('Y-m-d');
            $obj->sections = [
                'description' => __('The engine behind your AI-powered SaaS.', 'blyxxa-core'),
                'change_log' => $remote_data['changelog'] ?? __('See dashboard for details.', 'blyxxa-core')
            ];
            $obj->icons = [
                '1x' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif',
                '2x' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif',
                'default' => 'https://blyxxa.online/blyxxalabs/wp-content/uploads/2025/11/cropped-labs-logo-1-1.avif'
            ];
            $obj->banners = [];
            
            return $obj;
        }

        return $result;
    }
    
    public function clear_cache($upgrader, $options) {
        if ($options['action'] === 'update' && $options['type'] === 'plugin') {
            delete_transient('blyxxa_update_check');
        }
    }

    private function get_remote_version() {
        // Use cached check if available (1 hour)
        $cached = get_transient('blyxxa_update_check');
        if ($cached !== false) return $cached;

        if (empty($this->api_url)) return false;

        // No license required - all users can update
        $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $current_version = get_plugin_data($this->plugin_file)['Version'];

        $response = wp_remote_post($this->api_url . '/update-check', [
            'body' => wp_json_encode([
                'domain' => $domain,
                'current_version' => $current_version
            ]),
            'headers' => ['Content-Type' => 'application/json'],
            'timeout' => 15,
            'sslverify' => false
        ]);

        if (is_wp_error($response)) {
            // Cache failure for 5 minutes to avoid hammering
            set_transient('blyxxa_update_check', [], 5 * MINUTE_IN_SECONDS);
            return false;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['success']) && $body['success'] === true && !empty($body['data'])) {
            set_transient('blyxxa_update_check', $body['data'], HOUR_IN_SECONDS);
            return $body['data'];
        }

        // Cache empty result
        set_transient('blyxxa_update_check', [], 5 * MINUTE_IN_SECONDS);
        return false;
    }
}
