<?php
/**
 * Blyxxa Client Ping System
 * Sends periodic pings to Master Node for tracking
 * 
 * @package Blyxxa_Core
 */

if (!defined('ABSPATH')) exit;

/**
 * Send ping to Master Node
 * 
 * @param bool $is_active Whether the plugin is active
 * @return void
 */
function blyxxa_send_ping_to_master($is_active = true) {
    // Get Master API URL
    $api_url = defined('BLYXXA_MASTER_API_URL') 
        ? BLYXXA_MASTER_API_URL 
        : '';
    
    if (empty($api_url)) {
        return; // No API URL configured
    }
    
    // Build tracker ping URL (different namespace from license API)
    $ping_url = str_replace('/blyxxa/v1', '/blyxxa-tracker/v1', $api_url) . '/ping';
    
    // Prepare data package
    $data = [
        'site_url'    => home_url(),
        'version'     => defined('BLYXXA_VERSION') ? BLYXXA_VERSION : '0.0.0',
        'php_version' => phpversion(),
        'is_active'   => $is_active
    ];
    
    // Send non-blocking request (async)
    wp_remote_post($ping_url, [
        'body'      => wp_json_encode($data),
        'headers'   => ['Content-Type' => 'application/json'],
        'timeout'   => 10,
        'blocking'  => false // Async - don't wait for response
    ]);
}

/**
 * Send ping on plugin activation
 */
function blyxxa_send_activation_ping() {
    blyxxa_send_ping_to_master(true);
}

/**
 * Send weekly ping via cron job
 */
function blyxxa_send_weekly_ping() {
    blyxxa_send_ping_to_master(true);
}

/**
 * Schedule weekly ping event
 */
function blyxxa_schedule_weekly_ping() {
    if (!wp_next_scheduled('blyxxa_weekly_ping_event')) {
        wp_schedule_event(time(), 'weekly', 'blyxxa_weekly_ping_event');
    }
}

/**
 * Clear scheduled ping on deactivation
 */
function blyxxa_clear_weekly_ping_schedule() {
    $timestamp = wp_next_scheduled('blyxxa_weekly_ping_event');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'blyxxa_weekly_ping_event');
    }
}

/**
 * Send deactivation ping
 */
function blyxxa_send_deactivation_ping() {
    blyxxa_send_ping_to_master(false);
}

// Register hooks
add_action('init', 'blyxxa_schedule_weekly_ping');
add_action('blyxxa_weekly_ping_event', 'blyxxa_send_weekly_ping');
