<?php
// BLYXCORE SYSTEM (STABLE GLOBAL v2)

// =================================================================
// 1. WEBHOOK LISTENER V2 (Primary)
// =================================================================
add_action('rest_api_init', function () {
    // V2 Route (Current)
    register_rest_route('blyxxa/v1', '/webhook', array(
        'methods'  => ['POST', 'GET'], 
        'callback' => 'blyxxa_handle_webhook_v2', 
        'permission_callback' => '__return_true',
    ));
    
    // Legacy Route (For compatibility)
    register_rest_route('blyxxa/v1', '/webhook_legacy', array(
        'methods'  => ['POST', 'GET'], 
        'callback' => 'blyxxa_handle_webhook_legacy', 
        'permission_callback' => '__return_true',
    ));
});

// =================================================================
// 2. ROLE MANAGEMENT (Common Logic)
// =================================================================
add_action('init', function() {
    add_role('blyxxa_individual', 'Blyxxa Individual', array('read' => true));
    add_role('blyxxa_corporate', 'Blyxxa Corporate', array('read' => true));
    
    if (is_user_logged_in()) {
        $uid = get_current_user_id();
        $expire = get_user_meta($uid, 'blyxxa_expire_date', true);
        if ($expire && time() > $expire) {
            $u = new WP_User($uid);
            if(in_array('blyxxa_individual', $u->roles) || in_array('blyxxa_corporate', $u->roles)) {
                $u->set_role('subscriber');
                delete_user_meta($uid, 'blyxxa_expire_date');
            }
        }
    }
});

// =================================================================
// 3. SIGNAL HANDLER V2 (Logic)
// =================================================================
function blyxxa_handle_webhook_v2($request) {
    $text_domain = 'blyxxa-core';
    
    // 1. LOGGING SETUP
    $upload_dir = wp_upload_dir();
    $log_file = $upload_dir['basedir'] . '/blyxxa_webhook_logs.txt';
    $timestamp = current_time('mysql');
    
    // BROWSER TEST (GET)
    if ($request->get_method() === 'GET') {
        return new WP_REST_Response([
            'status' => 'success', 
            'message' => __('Webhook Line Active!', 'blyxxa-core')
        ], 200);
    }

    $params = $request->get_json_params();
    
    // 2. LOG INCOMING DATA
    $log_entry = "[$timestamp] IP: " . $_SERVER['REMOTE_ADDR'] . "\nPayload: " . print_r($params, true) . "\n-------------------\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);

    // 3. SMART DATA EXTRACTION (Recursive Search)
    $email = blyxxa_find_in_array($params, 'customer_email') ?: blyxxa_find_in_array($params, 'email');
    $type = $params['type'] ?? 'unknown';
    $prod_name = blyxxa_find_in_array($params, 'product_name');
    $plan_name = blyxxa_find_in_array($params, 'plan_name');
    
    // Fallback for specific Payhip structures
    if (!$prod_name && isset($params['items'][0]['product_name'])) {
        $prod_name = $params['items'][0]['product_name'];
    }

    if (!$email) {
        file_put_contents($log_file, "[$timestamp] ERROR: Email missing\n", FILE_APPEND);
        return new WP_Error('no_email', __('Email missing', 'blyxxa-core'), array('status' => 400));
    }

    $user = get_user_by('email', $email);
    
    // 4. CREATE USER IF NOT EXISTS
    if (!$user && ($type === 'subscription.created' || $type === 'paid')) {
        $password = wp_generate_password(12, false);
        $user_id = wp_create_user($email, $password, $email);
        
        if (!is_wp_error($user_id)) {
            $user = get_user_by('id', $user_id);
            
            // Try to find name
            $fname = blyxxa_find_in_array($params, 'customer_first_name') ?: blyxxa_find_in_array($params, 'first_name');
            $lname = blyxxa_find_in_array($params, 'customer_last_name') ?: blyxxa_find_in_array($params, 'last_name');
            
            if($fname) {
                wp_update_user(['ID' => $user_id, 'first_name' => $fname, 'last_name' => $lname]);
            }
            
            $login_url = function_exists('um_get_core_page') ? um_get_core_page('login') : wp_login_url();
            $subject = __('Blyxxa Labs Access Credentials', 'blyxxa-core');
            // translators: %1$s: user email, %2$s: password, %3$s: login URL
            $message = sprintf(__('Login Details:\n\nUser: %1$s\nPassword: %2$s\nLink: %3$s', 'blyxxa-core'), $email, $password, $login_url);
            wp_mail($email, $subject, $message);
            
            file_put_contents($log_file, "[$timestamp] INFO: New user created ($email)\n", FILE_APPEND);
        } else {
            file_put_contents($log_file, "[$timestamp] ERROR: User creation failed - " . $user_id->get_error_message() . "\n", FILE_APPEND);
        }
    }

    if (!$user) return new WP_REST_Response(['msg' => 'User Error'], 200);

    // 5. PRODUCT IDENTIFICATION
    $incoming_product_id = blyxxa_find_in_array($params, 'product_id');
    $incoming_permalink = blyxxa_find_in_array($params, 'product_permalink'); // Support for friendly URLs
    
    $saas_product_id = get_option('blyxxa_gumroad_product_id');
    $plugin_product_id = get_option('blyxxa_gumroad_plugin_product_id');
    
    // CASE A: PLUGIN RENTAL (License Management)
    // Check ID OR Permalink match
    if ($incoming_product_id === $plugin_product_id || ($incoming_permalink && strpos($incoming_permalink, $plugin_product_id) !== false)) {
        $license_key = blyxxa_find_in_array($params, 'license_key');
        if (!$license_key) return new WP_REST_Response(['msg' => 'License Key Missing'], 200);
        
        $licenses = get_option('blyxxa_client_licenses', []);
        
        // Sale / Reactivation
        if ($type === 'sale' || $type === 'subscription.created' || $type === 'subscription.resumed' || $type === 'subscription.updated') {
            $licenses[$license_key] = [
                'email' => $email,
                'status' => 'active',
                'last_check' => $timestamp,
                'domain' => isset($licenses[$license_key]['domain']) ? $licenses[$license_key]['domain'] : 'pending'
            ];
            update_option('blyxxa_client_licenses', $licenses);
            file_put_contents($log_file, "[$timestamp] LICENSE: Activated ($license_key)\n", FILE_APPEND);
            return new WP_REST_Response(['msg' => 'License Activated'], 200);
        }
        // Cancellation / Expiry
        elseif (strpos($type, 'cancelled') !== false || strpos($type, 'refunded') !== false || $type === 'subscription.expired' || $type === 'subscription.payment_failed') {
            if (isset($licenses[$license_key])) {
                $licenses[$license_key]['status'] = 'inactive';
                update_option('blyxxa_client_licenses', $licenses);
                file_put_contents($log_file, "[$timestamp] LICENSE: Deactivated ($license_key)\n", FILE_APPEND);
            }
            return new WP_REST_Response(['msg' => 'License Deactivated'], 200);
        }
        
        return new WP_REST_Response(['msg' => 'Plugin Event Ignored'], 200);
    }

    // CASE B: SAAS MEMBERSHIP (Role Assignment - Existing Logic)
    // If product ID matches OR if permalink matches OR if no specific product ID is set (legacy behavior)
    if ($incoming_product_id === $saas_product_id || ($incoming_permalink && strpos($incoming_permalink, $saas_product_id) !== false) || empty($saas_product_id)) {
        
        $target_role = 'blyxxa_individual'; // Default
        
        // Check for Corporate keywords in product or plan name
        if ((stripos($plan_name, 'Corporate') !== false) || (stripos($prod_name, 'Corporate') !== false)) {
            $target_role = 'blyxxa_corporate';
        }
    
        // ACTION BASED ON EVENT TYPE
        if ($type === 'subscription.created' || $type === 'subscription.resumed' || $type === 'sale') {
            $user->set_role($target_role);
            delete_user_meta($user->ID, 'blyxxa_expire_date');
            file_put_contents($log_file, "[$timestamp] ACTION: Role set to $target_role (SaaS)\n", FILE_APPEND);
            return new WP_REST_Response(['msg' => 'SaaS Activated'], 200);
        }
        elseif (strpos($type, 'deleted') !== false || strpos($type, 'cancelled') !== false || strpos($type, 'refunded') !== false || $type === 'subscription.expired') {
            $user->set_role('subscriber');
            file_put_contents($log_file, "[$timestamp] ACTION: Downgraded to subscriber (SaaS)\n", FILE_APPEND);
            return new WP_REST_Response(['msg' => 'SaaS Deactivated'], 200);
        }
    }

    return new WP_REST_Response(['msg' => 'Ignored Event: ' . $type], 200);
}

// Helper: Recursive Array Search
if (!function_exists('blyxxa_find_in_array')) {
    function blyxxa_find_in_array($array, $key_to_find) {
        if (!is_array($array)) return null;
        if (isset($array[$key_to_find])) return $array[$key_to_find];
        foreach ($array as $value) {
            if (is_array($value)) {
                $result = blyxxa_find_in_array($value, $key_to_find);
                if ($result) return $result;
            }
        }
        return null;
    }
}

// =================================================================
// 3.1 SIGNAL HANDLER LEGACY (Backup)
// =================================================================
function blyxxa_handle_webhook_legacy($request) {
    $text_domain = 'blyxxa-core';
    
    // Gumroad sends data as POST parameters, not JSON body
    $params = $request->get_body_params();
    
    // Log incoming data for debugging
    $log_file = wp_upload_dir()['basedir'] . '/blyxxa_webhook_logs.txt';
    $log_entry = "[" . current_time('mysql') . "] " . print_r($params, true) . "\n-------------------\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);

    $resource_name = $params['resource_name'] ?? '';
    $email = sanitize_email($params['email'] ?? '');
    $product_id = $params['product_id'] ?? '';
    
    // Get configured Product ID
    $target_product_id = get_option('blyxxa_gumroad_product_id');

    if (!$email) return new WP_Error('no_email', __('Email missing', 'blyxxa-core'), array('status' => 400));
    
    // Verify Product ID matches
    if ($product_id !== $target_product_id) {
        return new WP_REST_Response(['msg' => 'Ignored: Wrong Product ID'], 200);
    }

    $user = get_user_by('email', $email);

    // CREATE USER IF NOT EXISTS (Only on Sale)
    if (!$user && $resource_name === 'sale') {
        $password = wp_generate_password(12, false);
        $user_id = wp_create_user($email, $password, $email);
        
        if (!is_wp_error($user_id)) {
            $user = get_user_by('id', $user_id);
            // Try to get name from custom fields if available, or just use email
            // Gumroad doesn't always send name in standard fields
            
            $login_url = function_exists('um_get_core_page') ? um_get_core_page('login') : wp_login_url();
            $subject = __('Blyxxa Labs Access Credentials', 'blyxxa-core');
            // translators: %1$s: user email, %2$s: password, %3$s: login URL
            $message = sprintf(__('Login Details:\n\nUser: %1$s\nPassword: %2$s\nLink: %3$s', 'blyxxa-core'), $email, $password, $login_url);
            wp_mail($email, $subject, $message);
        }
    }

    if (!$user) return new WP_REST_Response(['msg' => 'User Error'], 200);

    // HANDLE EVENTS
    if ($resource_name === 'sale') {
        // Activate
        $user->set_role('blyxxa_corporate');
        return new WP_REST_Response(['msg' => 'Subscription Activated'], 200);
    }
    elseif (in_array($resource_name, ['cancellation', 'refund', 'dispute'])) {
        // Deactivate
        $user->set_role('subscriber');
        return new WP_REST_Response(['msg' => 'Subscription Deactivated'], 200);
    }

    return new WP_REST_Response(['msg' => 'Ignored Event'], 200);
}

// =================================================================
// 4. CONTENT LOCKER [blyxxa_lock]
// =================================================================
add_shortcode('blyxxa_lock', function($atts, $content = null) {
    $text_domain = 'blyxxa-core';
    // 1. ADMIN BYPASS
    if (current_user_can('administrator')) return do_shortcode($content);

    // Links
    $login_url = function_exists('um_get_core_page') ? um_get_core_page('login') : wp_login_url();
    $upgrade_url = get_option('blyxxa_payment_link', '#'); // Dynamic Sales page

    // 2. LOGIN CHECK
    if (!is_user_logged_in()) {
        return '<div style="background:#ffebeb; padding:20px; border:1px solid #ffa8a8; border-radius:8px; color:#c53030; text-align:center; font-family:sans-serif;">
                 🚫 <strong>Access Restricted.</strong> You must <a href="'.$login_url.'" style="color:#c53030; font-weight:bold; text-decoration:underline;">Login</a> to use this strategic tool.
                 <br><small style="color:#666;">Join as a free member to unlock instant access.</small>
        </div>';
    }

    $user = wp_get_current_user();
    $roles = (array) $user->roles;
    // Usage: [blyxxa_lock role="individual"]
    $required_level = isset($atts['role']) ? $atts['role'] : 'member';
    $access_granted = false;

    // 3. HIERARCHY LOGIC
    switch ($required_level) {
        case 'member':
            $access_granted = true;
            break;
        case 'individual':
            if (in_array('blyxxa_individual', $roles) || in_array('blyxxa_premium', $roles)) {
                $access_granted = true;
            }
            break;
        case 'corporate': 
        case 'premium':
            if (in_array('blyxxa_premium', $roles) || in_array('blyxxa_corporate', $roles)) {
                $access_granted = true;
            }
            break;
        default:
            if (in_array($required_level, $roles)) {
                $access_granted = true;
            }
            break;
    }

    // 4. RESULT
    if ($access_granted) {
        return do_shortcode($content);
    } else {
        $plan_name = ucfirst($required_level);
        return '<div style="background:#fff8e1; padding:25px; border:1px solid #ffe082; border-radius:10px; color:#856404; text-align:center; box-shadow:0 4px 6px rgba(0,0,0,0.05); font-family:sans-serif;">
            <h3 style="margin:0 0 10px 0;">🔒 Exclusive Tool</h3>
            <p style="margin:0 0 15px 0;">This strategic asset is reserved for <strong>'.$plan_name.'</strong> members.</p>
            <a href="'.$upgrade_url.'" data-gumroad-overlay-checkout="true" style="background:#000; color:#fff; padding:10px 20px; text-decoration:none; border-radius:5px; font-weight:bold;">Upgrade Your Plan 🚀</a>
        </div>';
    }
});

// =================================================================
// 5. UNIVERSAL TOOL RUNNER (Registry)
// =================================================================
add_action('wp_ajax_blyxxa_universal_runner', 'blyxxa_handle_universal_tool');

function blyxxa_handle_universal_tool() {
    // Security Check
    check_ajax_referer('blyxxa_tool_nonce', 'nonce');
    
    if (!is_user_logged_in()) wp_send_json_error('Oturum açmanız gerekiyor.');
    $user_id = get_current_user_id();
    $tool_id = isset($_POST['tool_id']) ? sanitize_text_field(wp_unslash($_POST['tool_id'])) : '';
    $model   = isset($_POST['model']) ? sanitize_text_field(wp_unslash($_POST['model'])) : '';
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $raw_inputs = isset($_POST['inputs']) ? (array)wp_unslash($_POST['inputs']) : [];
    $inputs = array_map('sanitize_text_field', $raw_inputs);
    $prompt_template = "";
    switch ($tool_id) {
        case 'viral_hook':
            $prompt_template = "ROL: Sen viral içerik mimarısın. GÖREV: '{topic}' konusu hakkında 5 adet viral kanca (hook) cümlesi yaz. TON: {tone}. TEKNİK: '{trigger}' psikolojik tetikleyicisini kullan. DİL: Çıktıları İngilizce ver (veya kullanıcının girdiği dilde). FORMAT: Her cümleyi yeni satıra yaz ve başına emoji koy.";
            break;
        case 'seo_makale_yazari':
            $prompt_template = "ROL: Sen '{uzmanlik}' konusunda dünya çapında bir uzmansın. GÖREV: '{baslik}' başlığıyla harika bir '{tur}' kaleme al. HEDEF KİTLE: Profesyoneller ve meraklılar. TON: {ton}. UZUNLUK HEDEFİ: Yaklaşık {uzunluk} kelime. YAPI KURALLARI: 1. Dikkat çekici bir giriş yap (Hook). 2. Konuyu H2 ve H3 başlıklarıyla derinlemesine incele. 3. Okuyucuya uygulanabilir tavsiyeler ver. 4. 'Key Takeaways' (Önemli Çıkarımlar) bölümü ekle. EKSTRA: Yazının en sonuna, '{gorsel_stil}' stilinde bir yapay zeka görsel oluşturma promptu (İngilizce) ekle. FORMAT: HTML kullan (h2, h3, p, ul, li). Sadece içeriği ver, giriş/çıkış konuşması yapma.";
            break;
        case 'yt_script':
            $prompt_template = "GÖREV: '{topic}' hakkındaki YouTube videosu için izleyiciyi ilk 5 saniyede kilitleyecek bir giriş (Intro) yaz. TON: Enerjik ve {tone}. DİL: İngilizce.";
            break;
        default:
            wp_send_json_error("HATA: Tanımsız Araç ID'si ($tool_id)");
            exit;
    }
    foreach ($inputs as $key => $value) {
        $prompt_template = str_replace('{' . $key . '}', $value, $prompt_template);
    }
    if (!function_exists('blyxxa_universal_api_call')) {
        wp_send_json_error("Çekirdek fonksiyon (API Call) bulunamadı.");
    }
    $result = blyxxa_universal_api_call($model, $prompt_template);
    if ($result['success']) {
        wp_send_json_success($result['data']);
    } else {
        wp_send_json_error($result['msg']);
    }
}