<?php
// BLYXXA API HANDLER

add_action('wp_ajax_blyxxa_fetch_models', function() {
    $user_id = get_current_user_id();
    $g_key = get_user_meta($user_id, 'blyxxa_gemini_key', true);
    $o_key = get_user_meta($user_id, 'blyxxa_openai_key', true);
    $p_key = get_user_meta($user_id, 'blyxxa_perplexity_key', true);
    $a_key = get_user_meta($user_id, 'blyxxa_anthropic_key', true);
    $or_key = get_user_meta($user_id, 'blyxxa_openrouter_key', true);
    $gr_key = get_user_meta($user_id, 'blyxxa_groq_key', true);

    $data = [
        'google' => [], 
        'openai' => [], 
        'perplexity' => [], 
        'anthropic' => [], 
        'openrouter' => [],
        'groq' => []
    ];

    // 1. Google Gemini (Late 2025)
    if($g_key) {
        $data['google'] = [
            ['id' => 'gemini-3-flash-preview', 'name' => __('Gemini Fast', 'blyxxa-core')],
            ['id' => 'gemini-exp-1206',                'name' => __('Gemini Thinking', 'blyxxa-core')],
            ['id' => 'gemini-3-pro-preview', 'name' => __('Gemini Pro', 'blyxxa-core')],
        ];
    }

    // 2. OpenAI (Late 2025)
    if($o_key) {
        $data['openai'] = [
            ['id' => 'gpt-4.5-turbo', 'name' => __('ChatGPT Fast', 'blyxxa-core')],
            ['id' => 'gpt-5',         'name' => __('ChatGPT Go', 'blyxxa-core')]
        ];
    }

    // 3. Perplexity
    if($p_key) {
        $data['perplexity'] = [
            ['id' => 'llama-3.1-sonar-large-128k-online', 'name' => 'Sonar Large (Fast Search)'],
            ['id' => 'llama-3.1-sonar-huge-128k-online',  'name' => 'Sonar Huge (Pro Search)']
        ];
    }

    // 4. Anthropic (Claude)
    if($a_key) {
        $data['anthropic'] = [
            ['id' => 'claude-3-7-sonnet-20250219', 'name' => 'Claude Sonnet (Balanced)'],
            ['id' => 'claude-3-5-opus-20240620',   'name' => 'Claude Opus (Most Powerful)']
        ];
    }

    // 5. Groq (Fast)
    if($gr_key) {
        $data['groq'] = [
            ['id' => 'llama-3.3-70b-versatile', 'name' => 'Llama Versatile (Fastest)'],
            ['id' => 'mixtral-8x7b-32768',      'name' => 'Mixtral (Economy)']
        ];
    }

    // 6. OpenRouter (Free Models Only)
    // Transist key v3 to force cache refresh
    if($or_key) {
        $cache_or = get_transient('blyxxa_models_or_free_v3_' . md5($or_key));
        if($cache_or) { $data['openrouter'] = $cache_or; }
        else {
            $resp = wp_remote_get('https://openrouter.ai/api/v1/models', ['headers' => ['Authorization' => 'Bearer ' . $or_key]]);
            if(!is_wp_error($resp)) {
                $body = json_decode(wp_remote_retrieve_body($resp), true);
                if(isset($body['data'])) {
                    foreach($body['data'] as $m) {
                        // Filter: Free Only (prompt & completion price must be 0)
                        $p_prompt = isset($m['pricing']['prompt']) ? floatval($m['pricing']['prompt']) : -1;
                        $p_comp = isset($m['pricing']['completion']) ? floatval($m['pricing']['completion']) : -1;
                        
                        if($p_prompt == 0 && $p_comp == 0) {
                            // Fix Double Free Label
                            $clean_name = preg_replace('/\s*\(free\)\s*/i', '', $m['name']);
                            $clean_name = trim($clean_name);
                            
                            $data['openrouter'][] = [
                                'id' => $m['id'],
                                'name' => $clean_name . ' (Free)'
                            ];
                        }
                    }
                    if(!empty($data['openrouter'])) set_transient('blyxxa_models_or_free_v3_' . md5($or_key), $data['openrouter'], 12 * HOUR_IN_SECONDS);
                }
            }
        }
        // Fallback if fetch fails but key exists
        if(empty($data['openrouter'])) {
             $data['openrouter'] = [
                ['id' => 'google/gemini-flash-1.5', 'name' => 'Gemini Flash (Free)'],
                ['id' => 'meta-llama/llama-3.1-70b-instruct:free', 'name' => 'Llama 3.1 70B (Free)'],
                ['id' => 'liquid/lfm-40b', 'name' => 'Liquid LFM 40B (Free)']
            ];
        }
    }

    wp_send_json_success($data);
});

// B. Tool Execution Handler (Modular)
add_action('wp_ajax_blyxxa_run_tool', function() {
    check_ajax_referer('blyxxa_tool_nonce', 'nonce');
    
    // LICENSE CHECK (Lock Execution)
    // LICENSE CHECK (Lock Execution)
    if (class_exists('Blyxxa_Client_License') && Blyxxa_Client_License::get_license_status() !== 'active') {
        // Enforce Free Limit: Only the Oldest Tool is allowed (Matches UI Logic)
        if (!defined('BLYXXA_TOOLS_PATH')) {
            define('BLYXXA_TOOLS_PATH', WP_CONTENT_DIR . '/uploads/blyxxa-tools/');
        }
        
        // Check if custom path is defined in main plugin, otherwise fallback to uploads
        $tools_path = defined('BLYXXA_TOOLS_PATH') ? BLYXXA_TOOLS_PATH : WP_CONTENT_DIR . '/uploads/blyxxa-tools/';
        
        $all_ts = glob($tools_path . '*.php');
        if($all_ts && count($all_ts) > 1) {
             usort($all_ts, function($a, $b) { return filectime($a) - filectime($b); });
             $allowed_slug = basename($all_ts[0], '.php');
             
             // Extract requested tool slug (remove tool_ prefix if present, though POST sends pure slug usually)
             // Shortcode sends ID which is often slug.
             // Let's assume tool_id matches filename for now as per Manager.
             
             if ($tool_id !== $allowed_slug) {
                  wp_send_json_error(__('License Invalid. This tool is inactive due to plan limits.', 'blyxxa-core'));
             }
        }
    }
    
    $tool_id = sanitize_text_field($_POST['tool_id']);
    $model = sanitize_text_field($_POST['model']);
    $is_rev = isset($_POST['revision_mode']) && $_POST['revision_mode'] == '1';
    
    // Load Tool Config
    $lib = get_option('blyxxa_tool_library', []);
    if(!isset($lib[$tool_id])) wp_send_json_error('Tool not found in library.');
    
    $tool = $lib[$tool_id];
    
    // SUBSCRIBER ROLE CHECK (End-User Access)
    $tool_role = isset($tool['role']) && !empty($tool['role']) ? $tool['role'] : 'read';
    
    $allowed = false;
    $user = wp_get_current_user();
    $user_roles = (array) $user->roles;

    // 1. FREE / MEMBER ACCESS
    if ($tool_role === 'member' || $tool_role === 'read' || $tool_role === 'subscriber') {
        if (current_user_can('read')) {
            $allowed = true;
        }
    } 
    // 2. PREMIUM ACCESS
    elseif ($tool_role === 'premium') {
        if (in_array('blyxxa_individual', $user_roles) || in_array('blyxxa_corporate', $user_roles) || current_user_can('administrator')) {
            $allowed = true;
        }
    } 
    // 3. CORPORATE ACCESS
    elseif ($tool_role === 'corporate') {
         if (in_array('blyxxa_corporate', $user_roles) || current_user_can('administrator')) {
            $allowed = true;
        }
    } 
    // 4. FALLBACK (Standard WP Caps)
    else {
        if (current_user_can($tool_role)) {
            $allowed = true;
        }
    }

    if (!$allowed) {
        // Custom Error for Premium/Corporate
        if ($tool_role === 'premium' || $tool_role === 'corporate') {
             $upgrade_url = get_option('blyxxa_payment_link', '#');
             $msg = '<div style="background:#fff8e1; padding:25px; border:1px solid #ffe082; border-radius:10px; color:#856404; text-align:center; box-shadow:0 4px 6px rgba(0,0,0,0.05); font-family:sans-serif;">
            <h3 style="margin:0 0 10px 0;">🔒 Exclusive Tool</h3>
            <p style="margin:0 0 15px 0;">This strategic asset is reserved for <strong>Premium</strong> members.</p>
            <a href="'.$upgrade_url.'" target="_blank" style="background:#000; color:#fff; padding:10px 20px; text-decoration:none; border-radius:5px; font-weight:bold;">Upgrade Your Plan 🚀</a>
        </div>';
             wp_send_json_error($msg);
        } else {
             wp_send_json_error(__('Access Denied: You do not have permission to use this tool.', 'blyxxa-core'));
        }
    }

    $prompt_template = $tool['prompt'];
    
    // Construct Prompt
    if($is_rev) {
        $prev = stripslashes($_POST['previous_result']);
        $inst = sanitize_textarea_field($_POST['revision_instruction']);
        $final_prompt = "ORIGINAL CONTENT:\n$prev\n\nREVISION INSTRUCTION:\n$inst\n\nPlease rewrite the content following the instruction.";
    } else {
        $final_prompt = $prompt_template;
        // Replace Variables
        if(isset($tool['inputs']) && is_array($tool['inputs'])) {
            foreach($tool['inputs'] as $inp) {
                $key = $inp['name'];
                // Handle numeric prefix logic from template
                if(is_numeric(substr($key, 0, 1))) $key = 'val_' . $key;
                
                $val = '';
                if(isset($_POST[$key])) {
                    if(is_array($_POST[$key])) $val = implode(', ', array_map('sanitize_text_field', $_POST[$key]));
                    else $val = sanitize_textarea_field($_POST[$key]);
                }
                
                // Replace {var}
                $final_prompt = str_replace('{'.$inp['name'].'}', $val, $final_prompt);
                // Also try with prefix if mismatch
                $final_prompt = str_replace('{'.$key.'}', $val, $final_prompt);
            }
        }
    }
    
    // Call API
    $res = blyxxa_universal_api_call($model, $final_prompt);
    
    if($res['success']) {
        // $html = wpautop($res['data']); 
        // Send raw content for client-side Markdown parsing
        wp_send_json_success($res['data']);
    } else {
        wp_send_json_error($res['msg']);
    }
});

// C. AI Tool Architect (Smart Import - LOCAL SECURE)
add_action('wp_ajax_blyxxa_ai_import', function() {
    check_ajax_referer('blyx_import_ai', 'nonce');
    
    $desc = sanitize_textarea_field($_POST['desc']);
    $model = sanitize_text_field($_POST['model']);
    
    if(!$desc || !$model) wp_send_json_error(['message' => 'Missing parameters']);
    
    // 1. Get Master Prompt (Decrypted from Cache)
    if (!class_exists('Blyxxa_Client_License')) {
        require_once plugin_dir_path(__FILE__) . 'blyxxa-client-license.php';
    }
    
    $master_prompt = Blyxxa_Client_License::get_master_prompt();
    
    if (!$master_prompt) {
        wp_send_json_error(['message' => 'Master Prompt could not be retrieved. Please check your license status or internet connection.']);
    }
    
    // 2. Construct Prompt (Dynamic Logic)
    // Split the Master Prompt at the marker
    $prompt_parts = explode('// BU DOSYANIN ARAÇ BİLGİLERİ SONU', $master_prompt);
    
    if (count($prompt_parts) > 1) {
        // Use the part AFTER the marker (System Rules) as Reference Material
        $system_rules = trim($prompt_parts[1]);
        
        $full_prompt = "*** META INSTRUCTION FOR ARCHITECT AI ***\n";
        $full_prompt .= "You are an expert AI Tool Architect. Your task is to design a new AI tool configuration (JSON) based on the user's request below.\n\n";
        $full_prompt .= "USER REQUEST: \"$desc\"\n\n";
        $full_prompt .= "*** REFERENCE MATERIAL: BLYXXA CONSTITUTION ***\n";
        $full_prompt .= "The tool you design must strictly follow the style, tone, and formatting rules defined in the 'Blyxxa Constitution' below.\n";
        $full_prompt .= "IMPORTANT: For the 'prompt' field in your JSON output, do NOT just copy this Constitution verbatim.\n";
        $full_prompt .= "Instead, write a SPECIFIC system prompt for the requested tool that *incorporates* these principles (like 'Contextual Value', 'Dynamic Persona', 'No Fluff').\n";
        $full_prompt .= "The 'prompt' you write should be ready to use by the new tool to generate high-quality content.\n\n";
        $full_prompt .= "--- BEGIN CONSTITUTION ---\n";
        $full_prompt .= $system_rules . "\n";
        $full_prompt .= "--- END CONSTITUTION ---\n";
    } else {
        // Fallback if marker not found
        $full_prompt = "You are an AI Tool Architect. Create a tool based on this request:\n" . $desc . "\n\nReference Rules:\n" . $master_prompt;
    }
    
    // FORCE JSON OUTPUT
    $full_prompt .= "\n\n" . '*** CRITICAL INSTRUCTION ***' . "\n";
    $full_prompt .= 'You must output the result ONLY as a valid JSON object. Do not include any conversational text, markdown formatting, or explanations outside the JSON.' . "\n";
    $full_prompt .= 'Map your response to these 3 sections exactly:' . "\n";
    $full_prompt .= '1. Tool Configuration -> title, id, desc' . "\n";
    $full_prompt .= '2. User Inputs -> inputs array (Create 7-8 inputs as per Constitution Rule #19)' . "\n";
    $full_prompt .= '3. Tool Task (Dynamic Prompt) -> prompt (The specific system prompt for this tool)' . "\n\n";
    $full_prompt .= 'The JSON structure must be:' . "\n";
    $full_prompt .= '```json' . "\n";
    $full_prompt .= '{
      "title": "Tool Title",
      "id": "tool_id_snake_case",
      "desc": "Short description",
      "prompt": "The full system prompt for the tool (include all variables like {var})",
      "inputs": [
        {
          "type": "text|textarea|select|number|checkbox",
          "name": "variable_name",
          "label": "Label",
          "placeholder": "Placeholder",
          "options": "Option1, Option2" (only for select, comma separated)
        }
      ]
    }' . "\n" . '```';
    
    // 3. Call AI (Local)
    $res = blyxxa_universal_api_call($model, $full_prompt);
    
    if($res['success']) {
        // Clean JSON
        $json_str = $res['data'];
        
        // UTF-8 Clean (Fix for "No Raw Data" error)
        $json_str = mb_convert_encoding($json_str, 'UTF-8', 'UTF-8');
        
        // Extract JSON block if wrapped in markdown
        if(preg_match('/\{.*\}/s', $json_str, $matches)) {
            $json_str = $matches[0];
        }
        
        // Remove comments
        $json_clean = preg_replace('!/\*.*?\*/!s', '', $json_str);
        $json_clean = preg_replace('/\n\s*\/\/.*$/m', '', $json_clean);
        
        $decoded = json_decode($json_clean);
        
        // Fallback
        if($decoded === null) {
            $decoded = json_decode($json_str);
        }

        wp_send_json_success(['json' => $decoded, 'raw' => $json_str]);
    } else {
        wp_send_json_error(['message' => $res['msg']]);
    }
});

// A. API Error Sanitizer (User-Friendly Messages)
if (!function_exists('blyxxa_sanitize_api_error')) {
    function blyxxa_sanitize_api_error($raw_error) {
        // Quota / Rate Limit errors
        if (stripos($raw_error, 'quota') !== false || stripos($raw_error, 'rate limit') !== false || stripos($raw_error, 'rate_limit') !== false) {
            return __('API quota exceeded. Please try again later or check your API plan.', 'blyxxa-core');
        }
        // Model not found
        if (stripos($raw_error, 'not found') !== false || stripos($raw_error, 'does not exist') !== false) {
            return __('The selected AI model is currently unavailable. Please try another model.', 'blyxxa-core');
        }
        // Invalid API key
        if (stripos($raw_error, 'invalid') !== false && stripos($raw_error, 'key') !== false) {
            return __('Invalid API key. Please check your API settings.', 'blyxxa-core');
        }
        // Unauthorized
        if (stripos($raw_error, 'unauthorized') !== false || stripos($raw_error, '401') !== false) {
            return __('API authentication failed. Please verify your API key.', 'blyxxa-core');
        }
        // Generic fallback
        return __('An error occurred while processing your request. Please try again.', 'blyxxa-core');
    }
}

// B. Universal API Call
if (!function_exists('blyxxa_universal_api_call')) {
    function blyxxa_universal_api_call($model, $prompt) {
        $user_id = get_current_user_id();
        $text_domain = 'blyxxa-core';

        $g_key = get_user_meta($user_id, 'blyxxa_gemini_key', true);
        $o_key = get_user_meta($user_id, 'blyxxa_openai_key', true);
        $p_key = get_user_meta($user_id, 'blyxxa_perplexity_key', true);
        $a_key = get_user_meta($user_id, 'blyxxa_anthropic_key', true);
        $or_key = get_user_meta($user_id, 'blyxxa_openrouter_key', true);
        $gr_key = get_user_meta($user_id, 'blyxxa_groq_key', true);

        $payload = [];
        $headers = ['Content-Type' => 'application/json'];
        $url = '';

        // 1. Google Gemini
        if (strpos($model, 'gemini') !== false && strpos($model, 'openrouter') === false) {
            if(!$g_key) return ['success'=>false, 'msg'=>'Google Gemini API Key Missing'];
            
            // Gemini 3 Preview usually requires v1beta. Gemini 2.0 Flash is GA on v1 but safe to use v1beta for all.
            $api_version = 'v1beta'; 
            $url = "https://generativelanguage.googleapis.com/$api_version/models/$model:generateContent?key=$g_key";
            
            $payload = ["contents" => [["parts" => [["text" => $prompt]]]]];
        }
        // 2. Anthropic Claude
        elseif (strpos($model, 'claude') !== false && strpos($model, 'openrouter') === false) {
            if(!$a_key) return ['success'=>false, 'msg'=>'Anthropic API Key Missing'];
            $url = 'https://api.anthropic.com/v1/messages';
            $headers['x-api-key'] = $a_key;
            $headers['anthropic-version'] = '2023-06-01';
            $payload = [
                "model" => $model,
                "max_tokens" => 4000,
                "messages" => [["role" => "user", "content" => $prompt]]
            ];
        }
        // 3. Perplexity
        elseif (strpos($model, 'sonar') !== false) {
            if(!$p_key) return ['success'=>false, 'msg'=>'Perplexity API Key Missing'];
            $url = 'https://api.perplexity.ai/chat/completions';
            $headers['Authorization'] = 'Bearer ' . $p_key;
            $payload = [
                "model" => $model,
                "messages" => [
                    ["role" => "system", "content" => "Be precise."],
                    ["role" => "user", "content" => $prompt]
                ]
            ];
        }
        // 4. Groq
        elseif ((in_array($model, ['llama-3.3-70b-versatile', 'llama-3.1-8b-instant', 'mixtral-8x7b-32768', 'gemma2-9b-it']) || (strpos($model, 'llama') !== false && $gr_key && !$or_key)) && strpos($model, '/') === false) {
            if(!$gr_key) return ['success'=>false, 'msg'=>'Groq API Key Missing'];
            $url = 'https://api.groq.com/openai/v1/chat/completions';
            $headers['Authorization'] = 'Bearer ' . $gr_key;
            $payload = [
                "model" => $model,
                "messages" => [["role" => "user", "content" => $prompt]]
            ];
        }
        // 5. OpenRouter (Catch-all for others or explicit OpenRouter models)
        elseif (strpos($model, '/') !== false || $or_key) {
            if(!$or_key) return ['success'=>false, 'msg'=>'OpenRouter API Key Missing'];
            $url = 'https://openrouter.ai/api/v1/chat/completions';
            $headers['Authorization'] = 'Bearer ' . $or_key;
            $headers['HTTP-Referer'] = site_url(); 
            $headers['X-Title'] = 'Blyxxa Tool';
            $payload = [
                "model" => $model,
                "messages" => [["role" => "user", "content" => $prompt]]
            ];
        }
        // 6. OpenAI (Default fallback for gpt-*)
        else {
            if(!$o_key) return ['success'=>false, 'msg'=>'OpenAI API Key Missing'];
            $url = 'https://api.openai.com/v1/chat/completions';
            $headers['Authorization'] = 'Bearer ' . $o_key;
            $payload = [
                "model" => $model,
                "messages" => [["role" => "user", "content" => $prompt]]
            ];
        }

        // EXECUTE REQUEST
        $args = [
            'body' => json_encode($payload),
            'headers' => $headers,
            'timeout' => 60
        ];

        // Log Request Details
        $log_msg = "API REQUEST:\nModel: $model\nURL: " . preg_replace('/key=([^&]+)/', 'key=***', $url) . "\nPayload: " . json_encode($payload) . "\n";
        file_put_contents(wp_upload_dir()['basedir'] . '/blyxxa_debug_log.txt', $log_msg, FILE_APPEND);

        // Handle GET vs POST (Gemini uses POST but URL params for key)
        $response = wp_remote_post($url, $args);

        if(is_wp_error($response)) {
            file_put_contents(wp_upload_dir()['basedir'] . '/blyxxa_debug_log.txt', "API ERROR: " . $response->get_error_message() . "\n", FILE_APPEND);
            return ['success'=>false, 'msg'=>$response->get_error_message()];
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_msg = wp_remote_retrieve_response_message($response);
        $raw_body = wp_remote_retrieve_body($response);
        
        file_put_contents(wp_upload_dir()['basedir'] . '/blyxxa_debug_log.txt', "API RESPONSE CODE: $response_code ($response_msg)\nRAW BODY:\n" . $raw_body . "\n----------------\n", FILE_APPEND);

        $body = json_decode($raw_body, true);
        
        // PARSE RESPONSE
        $content = '';
        if(isset($body['error'])) {
            $raw_err = $body['error']['message'] ?? json_encode($body['error']);
            return ['success'=>false, 'msg'=>blyxxa_sanitize_api_error($raw_err)];
        }
        
        // Gemini
        if(isset($body['candidates'][0]['content']['parts'][0]['text'])) {
            $content = $body['candidates'][0]['content']['parts'][0]['text'];
        }
        // Claude
        elseif(isset($body['content'][0]['text'])) {
            $content = $body['content'][0]['text'];
        }
        // OpenAI / OpenRouter / Groq / Perplexity
        elseif(isset($body['choices'][0]['message']['content'])) {
            $content = $body['choices'][0]['message']['content'];
        }
        
        file_put_contents(wp_upload_dir()['basedir'] . '/blyxxa_debug_log.txt', "PARSED CONTENT:\n" . $content . "\n----------------\n", FILE_APPEND);

        return ['success'=>true, 'data'=>$content];
    }
}