<?php
// Simple PO to MO compiler
function blyxxa_compile_po_to_mo($po_file, $mo_file) {
    if (!file_exists($po_file)) {
        die("Error: PO file not found: " . $po_file . "\n");
    }

    $lines = file($po_file, FILE_IGNORE_NEW_LINES);
    $strings = [];
    $current_msgid = '';
    $current_msgstr = '';
    $in_msgid = false;
    $in_msgstr = false;

    foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line) || strpos($line, '#') === 0) continue;

        if (strpos($line, 'msgid "') === 0) {
            if ($current_msgid !== '') {
                $strings[$current_msgid] = $current_msgstr;
            }
            $current_msgid = substr($line, 7, -1);
            $current_msgstr = '';
            $in_msgid = true;
            $in_msgstr = false;
        } elseif (strpos($line, 'msgstr "') === 0) {
            $current_msgstr = substr($line, 8, -1);
            $in_msgid = false;
            $in_msgstr = true;
        } elseif ($in_msgid && strpos($line, '"') === 0) {
            $current_msgid .= substr($line, 1, -1);
        } elseif ($in_msgstr && strpos($line, '"') === 0) {
            $current_msgstr .= substr($line, 1, -1);
        }
    }
    // Add last entry
    if ($current_msgid !== '') {
        $strings[$current_msgid] = $current_msgstr;
    }

    // Write MO file
    $output = fopen($mo_file, 'wb');
    
    // Magic number
    fwrite($output, pack('L', 0x950412de));
    // Revision
    fwrite($output, pack('L', 0));
    // Total strings
    fwrite($output, pack('L', count($strings)));
    // Offset to original strings table
    fwrite($output, pack('L', 28));
    // Offset to translated strings table
    fwrite($output, pack('L', 28 + (count($strings) * 8)));
    // Size of hash table (0 for now)
    fwrite($output, pack('L', 0));
    // Offset to hash table
    fwrite($output, pack('L', 28 + (count($strings) * 16)));

    // Write string tables
    $originals_table = '';
    $translations_table = '';
    $originals_data = '';
    $translations_data = '';
    
    $data_offset = 28 + (count($strings) * 16);

    foreach ($strings as $orig => $trans) {
        // Original
        $originals_table .= pack('L', strlen($orig));
        $originals_table .= pack('L', $data_offset + strlen($originals_data));
        $originals_data .= $orig . "\0";

        // Translation
        $translations_table .= pack('L', strlen($trans));
        $translations_table .= pack('L', $data_offset + strlen($originals_data) + strlen($translations_data)); // This logic is slightly flawed for combined writing, fixing below
    }
    
    // Re-calculate offsets correctly
    $originals_data = '';
    $translations_data = '';
    $originals_table = '';
    $translations_table = '';
    
    // We need to write tables first, then data.
    // Calculate total size of tables
    $table_size = count($strings) * 8; // 2 longs per entry
    
    $start_of_originals_data = 28 + (count($strings) * 16);
    
    // First pass to build data and calculate offsets
    $current_orig_offset = $start_of_originals_data;
    
    foreach ($strings as $orig => $trans) {
        $originals_data .= $orig . "\0";
    }
    
    $start_of_trans_data = $start_of_originals_data + strlen($originals_data);
    $current_trans_offset = $start_of_trans_data;
    
    foreach ($strings as $orig => $trans) {
        $translations_data .= $trans . "\0";
    }

    // Second pass to build tables
    $current_orig_running_offset = $start_of_originals_data;
    $current_trans_running_offset = $start_of_trans_data;

    foreach ($strings as $orig => $trans) {
        fwrite($output, pack('L', strlen($orig)));
        fwrite($output, pack('L', $current_orig_running_offset));
        $current_orig_running_offset += strlen($orig) + 1;
    }

    foreach ($strings as $orig => $trans) {
        fwrite($output, pack('L', strlen($trans)));
        fwrite($output, pack('L', $current_trans_running_offset));
        $current_trans_running_offset += strlen($trans) + 1;
    }

    // Write data
    fwrite($output, $originals_data);
    fwrite($output, $translations_data);

    fclose($output);
    echo "Compiled " . basename($po_file) . " to " . basename($mo_file) . "\n";
}

$blyxxa_lang_dir = dirname(__FILE__) . '/languages';
$blyxxa_po_files = glob($blyxxa_lang_dir . '/*.po');

foreach ($blyxxa_po_files as $blyxxa_po_file) {
    $blyxxa_mo_file = str_replace('.po', '.mo', $blyxxa_po_file);
    blyxxa_compile_po_to_mo($blyxxa_po_file, $blyxxa_mo_file);
}
?>
